<?php

use Illuminate\Auth\Events\Lockout;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\RateLimiter;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;
use Livewire\Attributes\Layout;
use Livewire\Attributes\Validate;
use Livewire\Volt\Component;

new class extends Component {
    #[Validate('required|string|min:3|max:50')]
    public string $username = '';

    #[Validate('required|string|min:4')]
    public string $password = '';

    public bool $remember = false;

    public $activeModal = null;

    public function mount()
    {
        $this->activeModal = null;
    }

    public function openLoginModal()
    {
        $this->activeModal = 'login';
        \Log::info('Login Modal Opened', ['activeModal' => $this->activeModal]);
    }

    public function openModal()
    {
        $this->openLoginModal();
    }

    #[\Livewire\Attributes\On('modal-switch')]
    public function handleModalSwitch($data)
    {
        if (isset($data['target']) && $data['target'] === 'login' && isset($data['key']) && isset($data['source'])) {
            $validSources = ['register', 'forgot_password'];
            if (in_array($data['source'], $validSources)) {
                $sessionData = session()->get($data['key']);
                if ($sessionData && $sessionData['action'] === 'open_login') {
                    session()->forget($data['key']);
                    $this->openLoginModal();
                }
            }
        }
    }

    public function switchToRegister()
    {
        $switchKey = 'modal_switch_' . uniqid();
        session()->put($switchKey, [
            'action' => 'open_register',
            'from' => 'login',
            'timestamp' => now()->timestamp,
        ]);

        $this->activeModal = null;
        $this->reset(['username', 'password', 'remember']);
        $this->resetErrorBag();

        $this->dispatch('modal-switch', [
            'target' => 'register',
            'key' => $switchKey,
            'source' => 'login',
        ])->to('auth.register');
    }

    public function switchToForgotPassword()
    {
        $switchKey = 'modal_switch_' . uniqid();
        session()->put($switchKey, [
            'action' => 'open_forgot_password',
            'from' => 'login',
            'timestamp' => now()->timestamp,
        ]);

        $this->activeModal = null;
        $this->reset(['username', 'password', 'remember']);
        $this->resetErrorBag();

        $this->dispatch('modal-switch', [
            'target' => 'forgot-password',
            'key' => $switchKey,
            'source' => 'login',
        ])->to('auth.forgot-password');
    }

    public function closeModal()
    {
        $this->activeModal = null;
        $this->reset(['username', 'password', 'remember']);
        $this->resetErrorBag();
        \Log::info('Login Modal Closed');
    }

    public function clearErrors()
    {
        $this->resetErrorBag();
    }
    /**
     * Handle an incoming authentication request.
     */
    public function login(): void
    {
        $this->validate();

        $this->ensureIsNotRateLimited();

        // استخدام Laravel Auth للمصادقة
        $credentials = [
            'username' => $this->username,
            'password' => $this->password,
        ];

        if (!Auth::attempt($credentials, $this->remember)) {
            RateLimiter::hit($this->throttleKey());

            throw ValidationException::withMessages([
                'username' => trans('auth.failed'),
            ]);
        }

        // التحقق من نجاح تسجيل الدخول
        if (!Auth::check()) {
            throw ValidationException::withMessages([
                'username' => trans('auth.login_failed'),
            ]);
        }

        RateLimiter::clear($this->throttleKey());

        // إعادة توليد الجلسة لأمان إضافي
        session()->regenerate();

        //$this->redirectIntended(default: route('home', absolute: false), navigate: true);

        // Show success notification and wait before reloading
        $this->dispatch('show-notification', [
            'type' => 'success',
            'message' => 'trans:auth.login_success',
            'duration' => 2500,
            'params' => [
                'username' => Auth::user()->name ?? Auth::user()->username
            ]
        ]);

        $this->closeModal();

        // Wait for the notification to be visible before reloading
        $this->js('setTimeout(function() { window.location.reload(); }, 2000);');
    } /**
     * Ensure the authentication request is not rate limited.
     */
    protected function ensureIsNotRateLimited(): void
    {
        if (!RateLimiter::tooManyAttempts($this->throttleKey(), 5)) {
            return;
        }

        event(new Lockout(request()));

        $seconds = RateLimiter::availableIn($this->throttleKey());

        throw ValidationException::withMessages([
            'username' => __('auth.throttle', [
                'seconds' => $seconds,
                'minutes' => ceil($seconds / 60),
            ]),
        ]);
    }

    /**
     * Get the authentication rate limiting throttle key.
     */
    protected function throttleKey(): string
    {
        return Str::transliterate(Str::lower($this->username) . '|' . request()->ip());
    }
}; ?>

<div>
    <!-- Login Modal Trigger Button -->
    <button class="auth-btn auth-btn-login hover:scale-105 transform transition-all duration-300 w-full lg:w-auto"
        wire:click="openLoginModal">
        <i class="fas fa-sign-in-alt mr-2 rtl:mr-0 rtl:ml-2"></i>
        <span wire:ignore  data-translate="authentication.login">Login</span>
    </button>

    <!-- Login Modal -->
    @if ($activeModal === 'login')
        <div x-data="{
            show: true,
            switching: false,
            modalState: 'normal',
            preventDoubleSwitch() {
                if (this.switching) return false;
                this.switching = true;
                setTimeout(() => { this.switching = false; }, 1000);
                return true;
            },
            triggerShake() { this.modalState = 'processing'; },
            setSuccessState() { this.modalState = 'success'; },
            setErrorState() { this.modalState = 'error'; },
            resetState() { this.modalState = 'normal'; }
        }" x-show="show" x-transition:enter="transition ease-out duration-500"
            x-transition:enter-start="opacity-0" x-transition:enter-end="opacity-100"
            x-transition:leave="transition ease-in duration-300" x-transition:leave-start="opacity-100"
            x-transition:leave-end="opacity-0" x-on:click.away="$wire.closeModal()"
            x-on:auth-processing.window="triggerShake()" x-on:auth-success.window="setSuccessState()"
            x-on:auth-error.window="setErrorState()"
            x-on:close-modal-delayed.window="setTimeout(() => { $wire.closeModal(); }, 1500);"
            class="modal-overlay login-modal">

            <!-- Animated Glass Background -->
            <div class="absolute inset-0 overflow-hidden pointer-events-none">
                <div class="glass-orb glass-orb-1"></div>
                <div class="glass-orb glass-orb-2"></div>
                <div class="glass-orb glass-orb-3"></div>
            </div>

            <div x-transition:enter="transition ease-out duration-500 delay-150"
                x-transition:enter-start="opacity-0 scale-75 rotate-3"
                x-transition:enter-end="opacity-100 scale-100 rotate-0"
                x-transition:leave="transition ease-in duration-200"
                x-transition:leave-start="opacity-100 scale-100 rotate-0"
                x-transition:leave-end="opacity-0 scale-90 rotate-1" class="modal-container"
                onclick="event.stopPropagation()" @keyup.window="resetState()">

                <!-- Close Button -->
                <button @click="show = false; $wire.closeModal()" class="modal-close">
                    <i class="fas fa-times"></i>
                    <!-- Fallback if FontAwesome doesn't load -->
                    <span class="close-fallback">×</span>
                </button>

                <!-- Modal Header -->
                <div class="modal-header">
                    <!-- Icon -->
                    <div class="modal-icon">
                        <i class="fas fa-sign-in-alt"></i>
                    </div>
                    <!-- Title -->
                    <h2 class="modal-title" data-en="Welcome Back!" data-ar="مرحباً بعودتك!"
                        wire:ignore  data-translate="authentication.welcomeBack">
                        Welcome Back!
                    </h2>
                    <!-- Subtitle -->
                    <p class="modal-subtitle" data-en="Sign in to your account to continue your adventure"
                        data-ar="سجل الدخول إلى حسابك لمتابعة مغامرتك" wire:ignore  data-translate="authentication.signInToContinue">
                        Sign in to your account to continue your adventure
                    </p>
                </div>

                <!-- Modal Header -->

                <!-- Login Form -->
                <form wire:submit="login" class="space-y-6" @submit="triggerShake()">
                    <!-- Email/Username Field -->
                    <div class="form-group">
                        <label class="form-label" data-en="Username or Email"
                            data-ar="اسم المستخدم أو البريد الإلكتروني" wire:ignore  data-translate="authentication.usernameOrEmail">
                            Username or Email
                        </label>
                        <div class="relative">
                            <div
                                class="absolute inset-y-0 ltr:left-0 rtl:right-0 ltr:pl-3 rtl:pr-3 flex items-center pointer-events-none">
                                <i class="fas fa-user text-gray-400"></i>
                            </div>
                            <input wire:model="username" type="text" autocomplete="username"
                                class="glass-input w-full ltr:pl-10 rtl:pr-10 ltr:pr-4 rtl:pl-4 @error('username') border-red-500 @enderror"
                                placeholder="Enter your username or email"
                                data-en-placeholder="Enter your username or email"
                                data-ar-placeholder="أدخل اسم المستخدم أو البريد الإلكتروني"
                                wire:ignore  data-translate="authentication.enterUsernameOrEmail" required />
                        </div>
                        @error('username')
                            <span class="text-red-500 text-sm mt-1">{{ $message }}</span>
                        @enderror
                    </div>

                    <!-- Password Field -->
                    <div class="form-group">
                        <label class="form-label" data-en="Password" data-ar="كلمة المرور"
                            wire:ignore  data-translate="authentication.password">
                            Password
                        </label>
                        <div class="relative">
                            <div
                                class="absolute inset-y-0 ltr:left-0 rtl:right-0 ltr:pl-3 rtl:pr-3 flex items-center pointer-events-none">
                                <i class="fas fa-lock text-gray-400"></i>
                            </div>
                            <input wire:model="password" type="password" autocomplete="current-password"
                                class="glass-input w-full ltr:pl-10 rtl:pr-10 ltr:pr-4 rtl:pl-4 @error('password') border-red-500 @enderror"
                                placeholder="Enter your password" data-en-placeholder="Enter your password"
                                data-ar-placeholder="أدخل كلمة المرور" wire:ignore  data-translate="authentication.enterPassword"
                                required />
                        </div>
                        @error('password')
                            <span class="text-red-500 text-sm mt-1">{{ $message }}</span>
                        @enderror
                    </div>

                    <!-- Remember Me & Forgot Password -->
                    <div class="flex items-center justify-between text-sm mt-6">
                        <label class="flex items-center cursor-pointer">
                            <input wire:model="remember" type="checkbox"
                                class="mr-2 rtl:mr-0 rtl:ml-2 rounded border-gray-600 bg-gray-700 text-purple-600 focus:ring-purple-500" />
                            <span class="text-gray-300" data-en="Remember me" data-ar="تذكرني"
                                wire:ignore  data-translate="authentication.rememberMe">Remember me</span>
                        </label>
                        <button type="button" @click="if(preventDoubleSwitch()) $wire.switchToForgotPassword()"
                            :disabled="switching"
                            class="text-purple-400 hover:text-blue-400 transition-colors duration-200 bg-transparent border-none cursor-pointer disabled:opacity-50 font-medium hover:underline"
                            data-en="Forgot password?" data-ar="نسيت كلمة المرور؟"
                            wire:ignore  data-translate="authentication.forgotPassword">
                            <span x-text="switching ? (window.getTranslation ? window.getTranslation('authentication.switching') : 'Switching...') : (window.getTranslation ? window.getTranslation('authentication.forgotPassword') : 'Forgot password?')"
                                x-bind:data-switching-text="switching ? (window.getTranslation ? window.getTranslation('authentication.switching') : 'Switching...') : ''"></span>
                        </button>
                    </div>

                    <!-- Submit Button -->
                    <button type="submit" wire:loading.attr="disabled" @click="triggerShake()"
                        class="glass-button">
                        <span wire:loading.remove class="flex items-center space-x-2 rtl:space-x-reverse">
                            <i class="fas fa-sign-in-alt"></i>
                            <span data-en="Sign In" data-ar="تسجيل الدخول"
                                wire:ignore  data-translate="authentication.signIn">Sign In</span>
                        </span>
                        <span wire:loading class="flex items-center justify-center space-x-2 rtl:space-x-reverse">
                            <i class="fas fa-spinner fa-spin"></i>
                            <span data-en="Signing in..." data-ar="جاري تسجيل الدخول..."
                                wire:ignore  data-translate="authentication.signingIn">Signing in...</span>
                        </span>
                    </button>
                </form>



                <!-- Footer -->
                <div class="form-footer">
                    <span class="inline-flex items-center space-x-1 rtl:space-x-reverse">
                        <span data-en="Don't have an account?" data-ar="لا تملك حساباً؟"
                            wire:ignore  data-translate="authentication.dontHaveAccount">
                            Don't have an account?
                        </span>
                        <button type="button" @click="if(preventDoubleSwitch()) $wire.switchToRegister()"
                            :disabled="switching"
                            class="text-purple-400 hover:text-blue-400 ltr:ml-1 rtl:mr-1 transition-all duration-200 bg-transparent border-none cursor-pointer disabled:opacity-50 font-medium hover:underline transform hover:scale-105"
                            data-en="Sign up" data-ar="إنشاء حساب" wire:ignore  data-translate="authentication.signUp">
                            <span x-text="switching ? (window.getTranslation ? window.getTranslation('authentication.switching') : 'Switching...') : (window.getTranslation ? window.getTranslation('authentication.signUp') : 'Sign up')"
                                x-bind:data-switching-text="switching ? (window.getTranslation ? window.getTranslation('authentication.switching') : 'Switching...') : ''"
                                wire:ignore  data-translate="authentication.switching"></span>
                        </button>
                    </span>
                </div>
            </div>
        </div>
    @endif
</div>
