<?php

use Livewire\Volt\Component;
use Livewire\Attributes\Vali        $this->dispatch('modal-switch', [
            'target' => 'login',
            'key' => $switchKey,
            'source' => 'forgot_password',
        ])->to('auth.login');
use Illuminate\Support\Facades\Password;
use Illuminate\Support\Facades\Session;

new class extends Component {
    #[Validate('required|email')]
    public string $email = '';

    public $activeModal = null;
    public $emailSent = false;
    public $message = '';

    public function mount()
    {
        $this->activeModal = null;
    }

    public function openForgotPasswordModal()
    {
        $this->activeModal = 'forgot-password';
        $this->emailSent = false;
        $this->message = '';
        \Log::info('Forgot Password Modal Opened', ['activeModal' => $this->activeModal]);
    }

    // Listen to modal switch from login
    #[\Livewire\Attributes\On('modal-switch')]
    public function handleModalSwitch($data)
    {
        if (isset($data['target']) && $data['target'] === 'forgot-password' &&
            isset($data['key']) && isset($data['source']) && $data['source'] === 'login') {

            // Verify session key to prevent duplicates
            $sessionData = session()->get($data['key']);
            if ($sessionData && $sessionData['action'] === 'open_forgot_password') {
                // Remove session key to prevent reuse
                session()->forget($data['key']);

                // Open forgot password modal safely
                $this->openForgotPasswordModal();
            }
        }
    }

    // Ultra-safe method to switch back to login modal
    public function switchToLogin()
    {
        // Generate unique session key
        $switchKey = 'modal_switch_' . uniqid();
        session()->put($switchKey, [
            'action' => 'open_login',
            'from' => 'forgot_password',
            'timestamp' => now()->timestamp
        ]);

        // Close this modal first
        $this->activeModal = null;
        $this->reset(['email', 'emailSent', 'message']);
        $this->resetErrorBag();

        // Use JavaScript for reliable switching
        $this->dispatch('modal-switch', [
            'target' => 'login',
            'key' => $switchKey,
            'source' => 'forgot_password'
        ])->to('auth.login');
    }

    public function closeModal()
    {
        $this->activeModal = null;
        $this->reset(['email', 'emailSent', 'message']);
        $this->resetErrorBag();
        \Log::info('Forgot Password Modal Closed');
    }

    public function sendPasswordResetLink()
    {
        $this->validate();

        // Send password reset link
        $status = Password::sendResetLink(['email' => $this->email]);

        if ($status === Password::RESET_LINK_SENT) {
            $this->emailSent = true;
            $this->message = 'We have emailed your password reset link!';
        } else {
            $this->addError('email', 'We can\'t find a user with that email address.');
        }
    }
};

?>

<div>

    <!-- Forgot Password Modal -->
    @if($activeModal === 'forgot-password')
        <div x-data="{
            show: true,
            switching: false,
            preventDoubleSwitch() {
                if (this.switching) return false;
                this.switching = true;
                setTimeout(() => { this.switching = false; }, 1000);
                return true;
            }
        }" x-show="show"
             x-transition:enter="transition ease-out duration-500"
             x-transition:enter-start="opacity-0"
             x-transition:enter-end="opacity-100"
             x-transition:leave="transition ease-in duration-300"
             x-transition:leave-start="opacity-100"
             x-transition:leave-end="opacity-0"
             @click.away="$wire.closeModal()"
             class="modal-overlay forgot-password-modal">

            <!-- Animated Glass Background -->
            <div class="absolute inset-0 overflow-hidden pointer-events-none">
                <div class="glass-orb glass-orb-1"></div>
                <div class="glass-orb glass-orb-2"></div>
                <div class="glass-orb glass-orb-3"></div>
            </div>

            <div x-transition:enter="transition ease-out duration-500 delay-150"
                 x-transition:enter-start="opacity-0 scale-75 rotate-3"
                 x-transition:enter-end="opacity-100 scale-100 rotate-0"
                 x-transition:leave="transition ease-in duration-200"
                 x-transition:leave-start="opacity-100 scale-100 rotate-0"
                 x-transition:leave-end="opacity-0 scale-90 rotate-1"
                 class="modal-container"
                 onclick="event.stopPropagation()">

                <!-- Close Button -->
                <button @click="show = false; $wire.closeModal()"
                    class="modal-close">
                    <i class="fas fa-times"></i>
                    <!-- Fallback if FontAwesome doesn't load -->
                    <span class="close-fallback">×</span>
                </button>

                <!-- Modal Header -->
                <div class="modal-header">
                    <!-- Icon -->
                    <div class="modal-icon">
                        <i class="fas fa-key"></i>
                    </div>
                    <!-- Title -->
                    <h2 class="modal-title"
                        data-en="Reset Password" data-ar="إعادة تعيين كلمة المرور">
                        Reset Password
                    </h2>
                    <!-- Subtitle -->
                    <p class="modal-subtitle"
                        data-en="Enter your email address and we'll send you a link to reset your password"
                        data-ar="أدخل عنوان بريدك الإلكتروني وسنرسل لك رابطاً لإعادة تعيين كلمة المرور">
                        Enter your email address and we'll send you a link to reset your password
                    </p>
                </div>

                @if($emailSent)
                    <!-- Success Message -->
                    <div class="success-message text-center p-6 mb-6">
                        <div class="flex justify-center mb-4">
                            <div class="w-16 h-16 bg-gradient-to-r from-green-500 to-emerald-600 rounded-full flex items-center justify-center shadow-lg"
                                 style="box-shadow: 0 0 30px rgba(34,197,94,0.6), 0 0 15px rgba(16,185,129,0.4), 0 8px 25px rgba(0,0,0,0.3);">
                                <i class="fas fa-check text-white text-2xl"
                                   style="text-shadow: 0 0 10px rgba(255,255,255,0.8), 0 0 5px rgba(34,197,94,0.5);"></i>
                            </div>
                        </div>
                        <h3 class="text-xl font-bold text-white mb-2" data-en="Email Sent!" data-ar="تم إرسال البريد الإلكتروني!">
                            Email Sent!
                        </h3>
                        <p class="text-gray-300 text-sm leading-relaxed" data-en="Check your email for the password reset link" data-ar="تحقق من بريدك الإلكتروني للحصول على رابط إعادة تعيين كلمة المرور">
                            Check your email for the password reset link
                        </p>

                        <!-- Back to Login Button -->
                        <button type="button"
                            @click="if(preventDoubleSwitch()) $wire.switchToLogin()"
                            :disabled="switching"
                            class="glass-button">
                            <span class="flex items-center space-x-2 rtl:space-x-reverse">
                                <i class="fas fa-arrow-left"></i>
                                <span data-en="Back to Login" data-ar="العودة لتسجيل الدخول">Back to Login</span>
                            </span>
                        </button>
                    </div>
                @else
                    <!-- Reset Password Form -->
                    <form wire:submit="sendPasswordResetLink" class="space-y-6">
                        <!-- Email Field -->
                        <div class="form-group">
                            <label class="form-label"
                                data-en="Email Address" data-ar="عنوان البريد الإلكتروني">
                                Email Address
                            </label>
                            <div class="relative">
                                <div class="absolute inset-y-0 ltr:left-0 rtl:right-0 ltr:pl-3 rtl:pr-3 flex items-center pointer-events-none">
                                    <i class="fas fa-envelope text-gray-400"></i>
                                </div>
                                <input wire:model="email" type="email" autocomplete="email"
                                    class="glass-input w-full ltr:pl-10 rtl:pr-10 ltr:pr-4 rtl:pl-4"
                                    placeholder="Enter your email address"
                                    data-en-placeholder="Enter your email address"
                                    data-ar-placeholder="أدخل عنوان بريدك الإلكتروني"
                                    required />
                            </div>
                            @error('email')
                                <p class="text-red-400 text-sm mt-2 rtl:text-right">{{ $message }}</p>
                            @enderror
                        </div>

                        <!-- Submit Button -->
                        <button type="submit" wire:loading.attr="disabled"
                            class="glass-button">
                            <span wire:loading.remove class="flex items-center space-x-2 rtl:space-x-reverse">
                                <i class="fas fa-paper-plane"></i>
                                <span data-en="Send Reset Link" data-ar="إرسال رابط إعادة التعيين">Send Reset Link</span>
                            </span>
                            <span wire:loading class="flex items-center justify-center space-x-2 rtl:space-x-reverse">
                                <i class="fas fa-spinner fa-spin"></i>
                                <span data-en="Sending..." data-ar="جاري الإرسال...">Sending...</span>
                            </span>
                        </button>
                    </form>

                    <!-- Footer -->
                    <div class="form-footer">
                        <span class="inline-flex items-center space-x-1 rtl:space-x-reverse">
                            <span data-en="Remember your password?" data-ar="تتذكر كلمة مرورك؟">
                                Remember your password?
                            </span>
                            <button type="button"
                                @click="if(preventDoubleSwitch()) $wire.switchToLogin()"
                                :disabled="switching"
                                class="text-purple-400 hover:text-blue-400 ltr:ml-1 rtl:mr-1 transition-all duration-200 bg-transparent border-none cursor-pointer disabled:opacity-50 font-medium hover:underline transform hover:scale-105"
                                data-en="Sign in" data-ar="تسجيل الدخول">
                                <span x-text="switching ? 'Switching...' : 'Sign in'"
                                    x-bind:data-switching-text="switching ? (document.documentElement.lang === 'ar' ? 'جاري التبديل...' : 'Switching...') : ''"></span>
                            </button>
                        </span>
                    </div>
                @endif
            </div>
        </div>
    @endif
</div>
