class SmartLanguageManager {
  constructor() {
    this.translations = {};
    this.currentLanguage = "en";
    this.loadedLanguages = new Set();
    this.showFallbackIndicators = false;
    this.isAuthenticated = false;
    this.userId = null;
    this.languageConfig = {};

    // Supported languages with their file paths
    this.supportedLanguages = {
      en: "/assets/languages/en.json",
      ar: "/assets/languages/ar.json",
      fr: "/assets/languages/fr.json",
      de: "/assets/languages/de.json",
      es: "/assets/languages/es.json",
      ru: "/assets/languages/ru.json",
      "zh-CN": "/assets/languages/zh-CN.json",
    };
  }

  /**
   * Initialize language system with server-side data
   */
  async init() {
    try {
      // Get language configuration from server
      this.languageConfig = window.languageConfig || {};
      this.isAuthenticated = this.languageConfig.isAuthenticated || false;
      this.userId = this.languageConfig.userId || null;

      // Determine initial language
      const initialLanguage = this.determineInitialLanguage();

      // Always load English first as fallback
      await this.loadLanguage("en");

      // Load and set the determined language
      if (initialLanguage !== "en") {
        await this.setLanguage(initialLanguage);
      } else {
        this.currentLanguage = "en";
        this.applyTranslations();
      }

      // Setup language switchers
      this.setupLanguageSwitcher();

      // Setup DOM observer for dynamically added elements
      this.setupDOMObserver();

      // Setup language preservation for dynamic content
      this.setupLanguagePreservation();

      // Language system initialized
    } catch (error) {
      // Language system initialization failed
      // Fallback to English
      this.currentLanguage = "en";
      this.setupLanguageSwitcher();
      this.showInitializationError();
    }
  }  /**
   * Determine initial language based on authentication status
   * @returns {string} Language code
   */
  determineInitialLanguage() {
    if (this.isAuthenticated) {
      // For authenticated users, use server-determined language
      const serverLang = this.languageConfig.currentLanguage;
      if (serverLang && this.isValidLanguage(serverLang)) {
        return serverLang;
      }
      return "en"; // Default to English if user's lang is invalid
    } else {
      // For non-authenticated users, try localStorage first, then browser detection
      const savedLang = localStorage.getItem('preferred-language');
      if (savedLang && this.isValidLanguage(savedLang)) {
        return savedLang;
      }

      // Use browser language or server-detected language
      const browserLang = this.languageConfig.browserLanguage || this.detectClientBrowserLanguage();
      return this.isValidLanguage(browserLang) ? browserLang : "en";
    }
  }

  /**
   * Detect browser language on client side (fallback)
   * @returns {string} Language code
   */
  detectClientBrowserLanguage() {
    const browserLang = navigator.language || navigator.userLanguage || 'en';
    const normalizedLang = this.normalizeLangCode(browserLang);
    return this.isValidLanguage(normalizedLang) ? normalizedLang : 'en';
  }

  /**
   * Normalize language code
   * @param {string} langCode
   * @returns {string}
   */
  normalizeLangCode(langCode) {
    const code = langCode.toLowerCase();
    const langMap = {
      'ar': 'ar', 'ar-sa': 'ar', 'ar-eg': 'ar',
      'en': 'en', 'en-us': 'en', 'en-gb': 'en',
      'fr': 'fr', 'fr-fr': 'fr',
      'de': 'de', 'de-de': 'de',
      'es': 'es', 'es-es': 'es',
      'ru': 'ru', 'ru-ru': 'ru',
      'zh': 'zh-CN', 'zh-cn': 'zh-CN'
    };
    return langMap[code] || code;
  }

  /**
   * Check if language code is valid
   * @param {string} langCode
   * @returns {boolean}
   */
  isValidLanguage(langCode) {
    return Object.keys(this.supportedLanguages).includes(langCode);
  }

  /**
   * Load language file
   * @param {string} langCode
   * @returns {Promise<Object|null>}
   */
  async loadLanguage(langCode) {
    if (this.loadedLanguages.has(langCode)) {
      return this.translations[langCode];
    }

    if (!this.supportedLanguages[langCode]) {
      // Language not supported
      return null;
    }

    try {
      const filePath = this.supportedLanguages[langCode];
      let langData;

      // Try different loading methods based on protocol
      if (window.location.protocol === "file:") {
        langData = await this.loadJsonViaXHR(filePath);
      } else {
        const response = await fetch(filePath);
        if (!response.ok) {
          throw new Error(`HTTP ${response.status}: ${response.statusText}`);
        }
        langData = await response.json();
      }

      // Validate language data structure
      if (!langData.meta || !langData.meta.language) {
        throw new Error(`Invalid language file structure for ${langCode}`);
      }

      this.translations[langCode] = langData;
      this.loadedLanguages.add(langCode);

      return langData;
    } catch (error) {
      // Failed to load language

      // Show user-friendly error message
      this.showLanguageLoadError(langCode);

      // Fallback to English if available
      if (langCode !== "en" && this.supportedLanguages["en"]) {
        return await this.loadLanguage("en");
      }

      return null;
    }
  }

  /**
   * Set language and update server if authenticated
   * @param {string} langCode
   */
  async setLanguage(langCode) {
    if (!this.isValidLanguage(langCode)) {
      // Invalid language code
      return;
    }

    // Add loading state
    this.addLoadingState();

    try {
      // Load English as fallback if not already loaded
      if (langCode !== "en" && !this.loadedLanguages.has("en")) {
        await this.loadLanguage("en");
      }

      // Load target language if not already loaded
      if (!this.loadedLanguages.has(langCode)) {
        const langData = await this.loadLanguage(langCode);
        if (!langData) {
          this.removeLoadingState();
          return;
        }
      }

      const currentDir = document.documentElement.dir;
      this.currentLanguage = langCode;
      const langData = this.translations[langCode];

      if (langData) {
        const newDir = langData.meta.direction;

        // Update document direction and language
        await this.animateDirectionChange(currentDir, newDir, langData, langCode);

        // Apply translations with animation
        this.applyTranslationsWithAnimation();

        // Save preference based on authentication status
        if (this.isAuthenticated) {
          // Update server-side preference
          await this.updateServerLanguage(langCode);
        }
        // Always save to localStorage for consistency
        localStorage.setItem("preferred-language", langCode);

        // Preserve language globally
        window.currentLanguagePreserved = langCode;

        // Update language switcher
        this.updateLanguageSwitcherWithAnimation();

        // Success animations
        this.removeLoadingState();
        this.addSuccessAnimation();

        // Force immediate reapplication
        setTimeout(() => {
          this.forceApplyCurrentLanguage();
        }, 300);

        // Language changed
        // Emit a DOM event for other components (Livewire) to react and sync
        try {
          const ev = new CustomEvent('language-changed', { detail: { language: langCode } });
          document.dispatchEvent(ev);
        } catch (e) {
          // ignore
        }
      }
    } catch (error) {
      // Failed to set language
      this.removeLoadingState();
      this.showLanguageChangeError(langCode);
    }
  }  /**
   * Update language preference on server
   * @param {string} langCode
   */
  async updateServerLanguage(langCode) {
    if (!this.isAuthenticated) {
      return;
    }

    try {
      const response = await fetch('/api/language/update', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || '',
          'Accept': 'application/json'
        },
        body: JSON.stringify({ language: langCode })
      });

      const data = await response.json();

      if (!data.success) {
        throw new Error(data.message || 'Server update failed');
      }

      // Server language preference updated successfully
    } catch (error) {
      // Failed to update server language preference
      // Continue with client-side functionality even if server update fails
    }
  }

  /**
   * Show language loading error
   * @param {string} langCode
   */
  showLanguageLoadError(langCode) {
    this.showNotification(
      'Language Load Error',
      `Failed to load ${langCode} language files. Using fallback.`,
      'error'
    );
  }

  /**
   * Show language change error
   * @param {string} langCode
   */
  showLanguageChangeError(langCode) {
    this.showNotification(
      'Language Change Error',
      `Failed to change language to ${langCode}. Please try again.`,
      'error'
    );
  }

  /**
   * Show notification
   * @param {string} title
   * @param {string} message
   * @param {string} type
   */
  showNotification(title, message, type = 'info') {
    // Ensure DOM is ready before creating notifications
    if (!document.body) {
      // DOM not ready for notification
      return;
    }

    const notification = document.createElement('div');
    notification.className = `fixed top-4 right-4 p-4 rounded-lg shadow-lg z-50 max-w-sm ${
      type === 'error' ? 'bg-red-600' : 'bg-blue-600'
    } text-white`;

    notification.innerHTML = `
      <div class="flex justify-between items-start">
        <div>
          <h4 class="font-semibold">${title}</h4>
          <p class="text-sm mt-1">${message}</p>
        </div>
        <button onclick="this.parentElement.parentElement.remove()" class="ml-2 text-white hover:text-gray-200">
          <svg class="w-4 h-4" fill="currentColor" viewBox="0 0 20 20">
            <path fill-rule="evenodd" d="M4.293 4.293a1 1 0 011.414 0L10 8.586l4.293-4.293a1 1 0 111.414 1.414L11.414 10l4.293 4.293a1 1 0 01-1.414 1.414L10 11.414l-4.293 4.293a1 1 0 01-1.414-1.414L8.586 10 4.293 5.707a1 1 0 010-1.414z" clip-rule="evenodd"></path>
          </svg>
        </button>
      </div>
    `;

    document.body.appendChild(notification);

    // Auto remove after 5 seconds
    setTimeout(() => {
      if (notification.parentElement) {
        notification.remove();
      }
    }, 5000);
  }

  // Copy all existing methods from MultiLanguageManager
  // (keeping the same implementation for compatibility)

  addLoadingState() {
    const switchers = document.querySelectorAll(".language-switcher, .language-switcher-mobile");
    switchers.forEach((switcher) => {
      if (switcher && switcher.classList) {
        switcher.classList.add("lang-loading");
      }
    });
  }

  removeLoadingState() {
    const switchers = document.querySelectorAll(".language-switcher, .language-switcher-mobile");
    switchers.forEach((switcher) => {
      if (switcher && switcher.classList) {
        switcher.classList.remove("lang-loading");
      }
    });
  }

  addSuccessAnimation() {
    const switchers = document.querySelectorAll(".language-switcher, .language-switcher-mobile");
    switchers.forEach((switcher) => {
      if (switcher && switcher.classList) {
        switcher.classList.add("lang-success");
        setTimeout(() => {
          if (switcher && switcher.classList) {
            switcher.classList.remove("lang-success");
          }
        }, 600);
      }
    });
  }

  async animateDirectionChange(currentDir, newDir, langData, langCode) {
    return new Promise((resolve) => {
      if (currentDir !== newDir) {
        // Ensure document.body exists before manipulating styles
        if (document.body) {
          document.body.style.transition = "all 0.3s cubic-bezier(0.4, 0, 0.2, 1)";
          document.body.style.opacity = "0.9";

          setTimeout(() => {
            document.documentElement.dir = newDir;
            document.documentElement.lang = langCode;

            setTimeout(() => {
              document.body.style.opacity = "1";
              resolve();
            }, 50);
          }, 150);
        } else {
          // If body doesn't exist, just update document element
          document.documentElement.dir = newDir;
          document.documentElement.lang = langCode;
          resolve();
        }
      } else {
        document.documentElement.lang = langCode;
        resolve();
      }
    });
  }

  applyTranslationsWithAnimation() {
    const elements = document.querySelectorAll("[data-translate]");

    elements.forEach((element, index) => {
      setTimeout(() => {
        const translationKey = element.getAttribute("data-translate");
        const translatedText = this.getTranslation(translationKey);

        element.style.transition = "opacity 0.2s ease";
        element.style.opacity = "0.7";

        setTimeout(() => {
          if (element.tagName === "INPUT" &&
              (element.type === "text" || element.type === "email" || element.type === "password")) {
            element.placeholder = translatedText;
          } else {
            element.textContent = translatedText;
          }

          element.style.opacity = "1";

          setTimeout(() => {
            element.style.transition = "";
          }, 200);
        }, 100);
      }, index * 20);
    });

    setTimeout(() => {
      this.updatePageTitle();
      this.updateMetaTags();
    }, elements.length * 20 + 100);
  }

  updateLanguageSwitcherWithAnimation() {
    const langData = this.translations[this.currentLanguage];
    if (!langData || !langData.meta) return;

    const updateElement = (element, newContent) => {
      if (!element) return;

      element.style.transition = "all 0.3s cubic-bezier(0.4, 0, 0.2, 1)";
      element.style.transform = "scale(0.9)";
      element.style.opacity = "0.7";

      setTimeout(() => {
        element.textContent = newContent;
        element.style.transform = "scale(1)";
        element.style.opacity = "1";

        setTimeout(() => {
          element.style.transition = "";
        }, 300);
      }, 150);
    };

    // Update all switchers
    const allFlags = document.querySelectorAll('[id$="-current-flag"]');
    const allLangs = document.querySelectorAll('[id$="-current-lang"]');

    allFlags.forEach(flag => updateElement(flag, langData.meta.flag));
    allLangs.forEach(lang => {
      const isMobile = lang.id.startsWith('mobile');
      updateElement(lang, isMobile ? langData.meta.language : langData.meta.code.toUpperCase());
    });
  }

  getTranslation(path) {
    const langData = this.translations[this.currentLanguage];
    if (!langData) {
      return this.getFallbackTranslation(path) || path;
    }

    const keys = path.split(".");
    let value = langData;

    for (const key of keys) {
      if (value && typeof value === "object" && key in value) {
        value = value[key];
      } else {
        return this.getFallbackTranslation(path) || path;
      }
    }

    return typeof value === "string" && value.trim() !== "" ? value : this.getFallbackTranslation(path) || path;
  }

  getFallbackTranslation(path) {
    if (this.currentLanguage === "en") {
      return null;
    }

    const englishData = this.translations["en"];
    if (!englishData) {
      return null;
    }

    const keys = path.split(".");
    let englishValue = englishData;

    for (const englishKey of keys) {
      if (englishValue && typeof englishValue === "object" && englishKey in englishValue) {
        englishValue = englishValue[englishKey];
      } else {
        return null;
      }
    }

    if (typeof englishValue === "string" && englishValue.trim() !== "") {
      return englishValue;
    }

    return null;
  }

  applyTranslations() {
    const elements = document.querySelectorAll("[data-translate]");

    elements.forEach((element) => {
      const translationKey = element.getAttribute("data-translate");
      const translatedText = this.getTranslation(translationKey);

      if (element.tagName === "INPUT" &&
          (element.type === "text" || element.type === "email" || element.type === "password")) {
        element.placeholder = translatedText;
      } else {
        element.textContent = translatedText;
      }
    });

    this.updatePageTitle();
    this.updateMetaTags();
  }

  updatePageTitle() {
    const titleElement = document.querySelector("title");
    if (titleElement) {
      const heroTitle = this.getTranslation("hero.title");
      const heroSubtitle = this.getTranslation("hero.subtitle");
      titleElement.textContent = `${heroTitle} - ${heroSubtitle}`;
    }
  }

  updateMetaTags() {
    const langData = this.translations[this.currentLanguage];
    if (langData && langData.meta) {
      document.documentElement.setAttribute("lang", langData.meta.code);
      document.documentElement.setAttribute("dir", langData.meta.direction);
    }
  }

  setupLanguageSwitcher() {
    // Create desktop/tablet language switchers
    const desktopSwitchers = document.querySelectorAll(".language-switcher");
    let switcherIndex = 0;
    desktopSwitchers.forEach(container => {
      this.createLanguageSwitcherForContainer(container, false, switcherIndex++);
    });

    // Create mobile language switcher
    this.createLanguageSwitcher(".language-switcher-mobile", true);

    setTimeout(() => {
      this.ensureAllDropdownsHiddenEdge();
    }, 100);
  }

  createLanguageSwitcherForContainer(container, isMobile = false, index = 0) {
    if (!container) {
      // Language switcher container not found
      return;
    }

    const availableLanguages = this.getAvailableLanguages();
    const uniqueId = isMobile ? "mobile" : `desktop-${index || 0}`;

    // Validate uniqueId to prevent invalid selectors
    if (!uniqueId || uniqueId.includes('undefined') || uniqueId.includes('null')) {
      // Invalid uniqueId for language switcher
      return;
    }

    // Create the switcher HTML
    const switcherHTML = `
      <div class="relative ${isMobile ? "w-full" : "inline-block"}">
        <button id="${uniqueId}-lang-btn"
                class="flex items-center space-x-2 px-3 py-2 text-sm font-medium text-gray-300 hover:text-white transition-colors duration-200 ${
                  isMobile ? "w-full justify-between" : ""
                }">
          <span class="flex items-center space-x-2">
            <span id="${uniqueId}-current-flag">🇺🇸</span>
            <span id="${uniqueId}-current-lang">${
      isMobile ? "English" : "EN"
    }</span>
          </span>
          <svg class="w-4 h-4 transform transition-transform duration-200" id="${uniqueId}-lang-chevron" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 9l-7 7-7-7"></path>
          </svg>
        </button>

        <div id="${uniqueId}-lang-dropdown"
             class="hidden absolute ${
               isMobile ? "left-0 right-0 top-full" : "right-0 top-full"
             } mt-2 py-2 bg-gray-800 rounded-lg shadow-xl border border-gray-700 z-50 ${
      isMobile ? "w-full" : "min-w-[160px]"
    }">
          ${availableLanguages
            .map(
              (lang) => `
            <button class="lang-option w-full px-4 py-2 text-left text-sm text-gray-300 hover:text-white hover:bg-gray-700 transition-colors duration-150 flex items-center space-x-3"
                    data-lang="${lang.code}">
              <span class="text-lg">${lang.flag}</span>
              <span>${lang.language}</span>
            </button>
          `
            )
            .join("")}
        </div>
      </div>
    `;

    container.innerHTML = switcherHTML;

    // Add event listeners using getElementById for better reliability
    const btn = document.getElementById(`${uniqueId}-lang-btn`);
    const dropdown = document.getElementById(`${uniqueId}-lang-dropdown`);
    const chevron = document.getElementById(`${uniqueId}-lang-chevron`);

    if (dropdown) {
      this.forceHideDropdownEdge(dropdown);
    }

    if (btn && dropdown) {
      // Toggle dropdown
      btn.addEventListener("click", (e) => {
        e.preventDefault();
        e.stopPropagation();
        const isHidden = dropdown && dropdown.classList ? dropdown.classList.contains("hidden") : true;

        // Close all other dropdowns
        document.querySelectorAll('[id$="-lang-dropdown"]').forEach((dd) => {
          if (dd && dd !== dropdown && dd.classList) {
            this.forceHideDropdownEdge(dd);
            dd.classList.add("hidden");
          }
        });

        if (isHidden) {
          this.forceShowDropdownEdge(dropdown);
          if (dropdown && dropdown.classList) {
            dropdown.classList.remove("hidden");
          }
          if (chevron) chevron.style.transform = "rotate(180deg)";
        } else {
          this.forceHideDropdownEdge(dropdown);
          if (dropdown && dropdown.classList) {
            dropdown.classList.add("hidden");
          }
          if (chevron) chevron.style.transform = "rotate(0deg)";
        }
      });

      // Language selection
      dropdown.addEventListener("click", async (e) => {
        const langOption = e.target.closest(".lang-option");
        if (langOption) {
          e.preventDefault();
          const selectedLang = langOption.getAttribute("data-lang");

          this.forceHideDropdownEdge(dropdown);
          if (dropdown && dropdown.classList) {
            dropdown.classList.add("hidden");
          }
          if (chevron) chevron.style.transform = "rotate(0deg)";

          await this.setLanguage(selectedLang);
        }
      });
    }

    // Close dropdown when clicking outside
    document.addEventListener("click", (e) => {
      if (!container.contains(e.target)) {
        this.forceHideDropdownEdge(dropdown);
        dropdown?.classList.add("hidden");
        if (chevron) chevron.style.transform = "rotate(0deg)";
      }
    });

    // Initial update
    this.updateLanguageSwitcher();
  }

  ensureAllDropdownsHiddenEdge() {
    const dropdowns = document.querySelectorAll('[id$="-lang-dropdown"]');
    dropdowns.forEach(dropdown => {
      this.forceHideDropdownEdge(dropdown);
    });

    const chevrons = document.querySelectorAll('[id$="-lang-chevron"]');
    chevrons.forEach(chevron => {
      if (chevron) {
        chevron.style.transform = "rotate(0deg)";
      }
    });
  }

  createLanguageSwitcher(selector, isMobile = false) {
    const container = document.querySelector(selector);
    if (!container) {
      return;
    }

    const availableLanguages = this.getAvailableLanguages();

    const switcherHTML = `
      <div class="relative ${isMobile ? "w-full" : "inline-block"}">
        <button id="${isMobile ? "mobile" : "desktop"}-lang-btn"
                class="flex items-center space-x-2 px-3 py-2 text-sm font-medium text-gray-300 hover:text-white transition-colors duration-200 ${
                  isMobile ? "w-full justify-between" : ""
                }">
          <span class="flex items-center space-x-2">
            <span id="${isMobile ? "mobile" : "desktop"}-current-flag">🇺🇸</span>
            <span id="${isMobile ? "mobile" : "desktop"}-current-lang">${
      isMobile ? "English" : "EN"
    }</span>
          </span>
          <svg class="w-4 h-4 transform transition-transform duration-200" id="${
            isMobile ? "mobile" : "desktop"
          }-lang-chevron" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 9l-7 7-7-7"></path>
          </svg>
        </button>

        <div id="${isMobile ? "mobile" : "desktop"}-lang-dropdown"
             class="hidden absolute ${
               isMobile ? "left-0 right-0 top-full" : "right-0 top-full"
             } mt-2 py-2 bg-gray-800 rounded-lg shadow-xl border border-gray-700 z-50 ${
      isMobile ? "w-full" : "min-w-[160px]"
    }">
          ${availableLanguages
            .map(
              (lang) => `
            <button class="lang-option w-full px-4 py-2 text-left text-sm text-gray-300 hover:text-white hover:bg-gray-700 transition-colors duration-150 flex items-center space-x-3"
                    data-lang="${lang.code}">
              <span class="text-lg">${lang.flag}</span>
              <span>${lang.language}</span>
            </button>
          `
            )
            .join("")}
        </div>
      </div>
    `;

    container.innerHTML = switcherHTML;

    const btn = document.getElementById(`${isMobile ? "mobile" : "desktop"}-lang-btn`);
    const dropdown = document.getElementById(`${isMobile ? "mobile" : "desktop"}-lang-dropdown`);
    const chevron = document.getElementById(`${isMobile ? "mobile" : "desktop"}-lang-chevron`);

    if (dropdown) {
      this.forceHideDropdownEdge(dropdown);
    }

    if (btn && dropdown) {
      btn.addEventListener("click", (e) => {
        e.preventDefault();
        e.stopPropagation();
        const isHidden = dropdown && dropdown.classList ? dropdown.classList.contains("hidden") : true;

        document.querySelectorAll('[id$="-lang-dropdown"]').forEach((dd) => {
          if (dd !== dropdown) {
            this.forceHideDropdownEdge(dd);
            if (dd && dd.classList) {
              dd.classList.add("hidden");
            }
          }
        });

        if (isHidden) {
          this.forceShowDropdownEdge(dropdown);
          if (dropdown && dropdown.classList) {
            dropdown.classList.remove("hidden");
          }
          if (chevron) chevron.style.transform = "rotate(180deg)";
        } else {
          this.forceHideDropdownEdge(dropdown);
          if (dropdown && dropdown.classList) {
            dropdown.classList.add("hidden");
          }
          if (chevron) chevron.style.transform = "rotate(0deg)";
        }
      });

      dropdown.addEventListener("click", async (e) => {
        const langOption = e.target.closest(".lang-option");
        if (langOption) {
          e.preventDefault();
          const selectedLang = langOption.getAttribute("data-lang");

          this.forceHideDropdownEdge(dropdown);
          if (dropdown && dropdown.classList) {
            dropdown.classList.add("hidden");
          }
          if (chevron) chevron.style.transform = "rotate(0deg)";

          await this.setLanguage(selectedLang);
        }
      });
    }

    document.addEventListener("click", (e) => {
      if (!container.contains(e.target)) {
        this.forceHideDropdownEdge(dropdown);
        dropdown?.classList.add("hidden");
        if (chevron) chevron.style.transform = "rotate(0deg)";
      }
    });

    this.updateLanguageSwitcher();
  }

  updateLanguageSwitcher() {
    this.updateLanguageSwitcherWithAnimation();
  }

  getAvailableLanguages() {
    return [
      { code: "en", language: "English", flag: "🇺🇸" },
      { code: "ar", language: "العربية", flag: "🇸🇦" },
      { code: "fr", language: "Français", flag: "🇫🇷" },
      { code: "de", language: "Deutsch", flag: "🇩🇪" },
      { code: "es", language: "Español", flag: "🇪🇸" },
      { code: "ru", language: "Русский", flag: "🇷🇺" },
      { code: "zh-CN", language: "简体中文", flag: "🇨🇳" },
    ];
  }

  getCurrentLanguageInfo() {
    const langData = this.translations[this.currentLanguage];
    return langData
      ? langData.meta
      : { language: "English", code: "en", direction: "ltr", flag: "🇺🇸" };
  }

  refreshTranslations() {
    this.applyTranslations();
  }

  translateElement(element) {
    if (!element || !element.hasAttribute('data-translate')) {
      return;
    }

    const translationKey = element.getAttribute("data-translate");
    const translatedText = this.getTranslation(translationKey);

    const isUsingFallback = this.currentLanguage !== "en" &&
                           (!this.translations[this.currentLanguage] ||
                            !this.getDirectTranslation(translationKey, this.currentLanguage));

    if (element.tagName === "INPUT" &&
        (element.type === "text" || element.type === "email" || element.type === "password")) {
      element.placeholder = translatedText;
    } else {
      element.textContent = translatedText;
    }

    if (isUsingFallback && this.showFallbackIndicators) {
      if (element && element.classList) {
        element.classList.add('fallback-translation');
      }
      element.title = `Translation not available in current language. Showing English version.`;
    } else {
      if (element && element.classList) {
        element.classList.remove('fallback-translation');
      }
      element.removeAttribute('title');
    }
  }

  getDirectTranslation(path, langCode) {
    const langData = this.translations[langCode];
    if (!langData) return null;

    const keys = path.split(".");
    let value = langData;

    for (const key of keys) {
      if (value && typeof value === "object" && key in value) {
        value = value[key];
      } else {
        return null;
      }
    }

    return typeof value === "string" && value.trim() !== "" ? value : null;
  }

  loadJsonViaXHR(filePath) {
    return new Promise((resolve, reject) => {
      const xhr = new XMLHttpRequest();
      xhr.open("GET", filePath, true);
      xhr.responseType = "text";

      xhr.onload = function () {
        if (xhr.status === 200) {
          try {
            const data = JSON.parse(xhr.responseText);
            resolve(data);
          } catch (e) {
            reject(new Error(`Invalid JSON in ${filePath}: ${e.message}`));
          }
        } else {
          reject(new Error(`HTTP ${xhr.status}: ${xhr.statusText}`));
        }
      };

      xhr.onerror = function () {
        reject(new Error(`Network error loading ${filePath}`));
      };

      xhr.send();
    });
  }

  showInitializationError() {
    this.showNotification(
      'Language System Error',
      'Failed to load language files. Using default text.',
      'error'
    );
  }

  setFallbackIndicators(enable) {
    this.showFallbackIndicators = enable;
    this.applyTranslations();
  }

  getTranslationStats() {
    const currentLang = this.currentLanguage;
    const englishData = this.translations["en"];
    const currentData = this.translations[currentLang];

    if (!englishData) {
      return { error: "English base language not loaded" };
    }

    if (currentLang === "en") {
      return {
        language: "en",
        totalKeys: this.countTranslationKeys(englishData),
        translatedKeys: this.countTranslationKeys(englishData),
        missingKeys: 0,
        coverage: 100
      };
    }

    if (!currentData) {
      return {
        language: currentLang,
        totalKeys: this.countTranslationKeys(englishData),
        translatedKeys: 0,
        missingKeys: this.countTranslationKeys(englishData),
        coverage: 0
      };
    }

    const totalKeys = this.countTranslationKeys(englishData);
    const translatedKeys = this.countTranslationKeys(currentData);
    const missingKeys = totalKeys - translatedKeys;
    const coverage = Math.round((translatedKeys / totalKeys) * 100);

    return {
      language: currentLang,
      totalKeys,
      translatedKeys,
      missingKeys,
      coverage
    };
  }

  forceHideDropdownEdge(dropdown) {
    if (!dropdown || !dropdown.classList) return;

    // Remove focus from any focused element inside the dropdown before hiding
    const focusedElement = dropdown.querySelector(':focus');
    if (focusedElement) {
      focusedElement.blur();
    }

    dropdown.classList.add('hidden');
    dropdown.style.display = 'none';
    dropdown.style.visibility = 'hidden';
    dropdown.style.opacity = '0';
    dropdown.style.pointerEvents = 'none';
    dropdown.style.height = '0';
    dropdown.style.overflow = 'hidden';
    dropdown.style.transform = 'translateY(-100%) scale(0.95)';

    // Add aria-hidden after a short delay to ensure focus is removed
    setTimeout(() => {
      dropdown.setAttribute('aria-hidden', 'true');
    }, 10);

    dropdown.offsetHeight;
  }

  forceShowDropdownEdge(dropdown) {
    if (!dropdown || !dropdown.classList) return;

    dropdown.classList.remove('hidden');
    dropdown.style.display = 'block';
    dropdown.style.visibility = 'visible';
    dropdown.style.opacity = '1';
    dropdown.style.pointerEvents = 'auto';
    dropdown.style.height = 'auto';
    dropdown.style.overflow = 'visible';
    dropdown.style.transform = 'translateY(0) scale(1)';
    dropdown.removeAttribute('aria-hidden');

    this.adjustDropdownPositionForEdge(dropdown, dropdown.previousElementSibling);

    dropdown.offsetHeight;
  }

  countTranslationKeys(langData) {
    let count = 0;

    const countRecursive = (obj) => {
      for (const key in obj) {
        if (typeof obj[key] === "string") {
          count++;
        } else if (typeof obj[key] === "object" && obj[key] !== null) {
          countRecursive(obj[key]);
        }
      }
    };

    const { meta, ...translations } = langData;
    countRecursive(translations);

    return count;
  }

  adjustDropdownPositionForEdge(dropdown, button) {
    if (!dropdown || !button || !dropdown.style) return;

    const buttonRect = button.getBoundingClientRect();
    const viewportHeight = window.innerHeight;
    const viewportWidth = window.innerWidth;
    const dropdownHeight = 200;
    const spaceBelow = viewportHeight - buttonRect.bottom;
    const spaceAbove = buttonRect.top;
    const buffer = 20;

    dropdown.style.removeProperty('top');
    dropdown.style.removeProperty('bottom');
    dropdown.style.removeProperty('transform');
    dropdown.style.removeProperty('maxHeight');
    dropdown.style.removeProperty('overflowY');

    if (spaceBelow >= dropdownHeight + buffer) {
      dropdown.style.top = 'calc(100% + 8px)';
      dropdown.style.bottom = 'auto';
      dropdown.style.maxHeight = `${Math.min(spaceBelow - buffer, 350)}px`;
    } else if (spaceAbove >= dropdownHeight + buffer) {
      dropdown.style.top = 'auto';
      dropdown.style.bottom = 'calc(100% + 8px)';
      dropdown.style.maxHeight = `${Math.min(spaceAbove - buffer, 350)}px`;
    } else {
      if (spaceBelow > spaceAbove) {
        dropdown.style.top = 'calc(100% + 8px)';
        dropdown.style.bottom = 'auto';
        dropdown.style.maxHeight = `${spaceBelow - buffer}px`;
      } else {
        dropdown.style.top = 'auto';
        dropdown.style.bottom = 'calc(100% + 8px)';
        dropdown.style.maxHeight = `${spaceAbove - buffer}px`;
      }
    }

    dropdown.style.overflowY = 'auto';
    dropdown.style.overflowX = 'hidden';

    const isRTL = document.documentElement.dir === 'rtl';
    const dropdownWidth = 160;

    if (isRTL) {
      dropdown.style.left = '0';
      dropdown.style.right = 'auto';
    } else {
      const spaceRight = viewportWidth - buttonRect.right;
      if (spaceRight < dropdownWidth) {
        dropdown.style.right = '0';
        dropdown.style.left = 'auto';
      } else {
        dropdown.style.right = '0';
        dropdown.style.left = 'auto';
      }
    }

    dropdown.style.zIndex = '2147483647';
    dropdown.style.position = 'absolute';
    dropdown.style.willChange = 'transform';
    dropdown.offsetHeight;
    dropdown.style.willChange = 'auto';
  }

  setupDOMObserver() {
    // Ensure document.body exists before setting up observer
    if (!document.body) {
      // DOM not ready for observer setup
      return;
    }

    const observer = new MutationObserver((mutations) => {
      let hasNewTranslatableElements = false;

      mutations.forEach((mutation) => {
        mutation.addedNodes.forEach((node) => {
          if (node.nodeType === Node.ELEMENT_NODE) {
            if (node.hasAttribute && node.hasAttribute('data-translate')) {
              hasNewTranslatableElements = true;
            } else if (node.querySelector) {
              const translatableElements = node.querySelectorAll('[data-translate]');
              if (translatableElements.length > 0) {
                hasNewTranslatableElements = true;
              }
            }
          }
        });
      });

      if (hasNewTranslatableElements) {
        setTimeout(() => {
          this.applyTranslations();
        }, 100);
      }
    });

    observer.observe(document.body, {
      childList: true,
      subtree: true
    });
  }

  /**
   * Setup language preservation for dynamic content updates
   */
  setupLanguagePreservation() {
    // Listen for Livewire updates
    if (window.Livewire) {
      // For Livewire v3
      window.Livewire.on('*', () => {
        this.debounceApplyTranslations();
      });

      // Hook into Livewire lifecycle events
      document.addEventListener('livewire:navigated', () => {
        this.debounceApplyTranslations();
      });

      document.addEventListener('livewire:load', () => {
        this.debounceApplyTranslations();
      });

      document.addEventListener('livewire:update', () => {
        this.debounceApplyTranslations();
      });
    }

    // Listen for Alpine.js updates
    document.addEventListener('alpine:init', () => {
      this.debounceApplyTranslations();
    });

    // Add Alpine.js magic property for $translate
    if (typeof window !== 'undefined') {
      // Define Alpine.js magic property immediately if Alpine is available
      if (window.Alpine && typeof window.Alpine.magic === 'function') {
        window.Alpine.magic('translate', () => {
          return (key, fallback = null) => {
            if (window.languageManager && typeof window.languageManager.getTranslation === 'function') {
              return window.languageManager.getTranslation(key);
            }
            // Fallback: return the fallback or key itself if language manager not available
            return fallback || key;
          };
        });
      }

      // Also listen for Alpine initialization to ensure the magic property is registered
      document.addEventListener('alpine:init', () => {
        if (window.Alpine && typeof window.Alpine.magic === 'function') {
          window.Alpine.magic('translate', () => {
            return (key, fallback = null) => {
              if (window.languageManager && typeof window.languageManager.getTranslation === 'function') {
                return window.languageManager.getTranslation(key);
              }
              // Fallback: return the fallback or key itself if language manager not available
              return fallback || key;
            };
          });
        }
      });
    }

    // Listen for custom preservation events
    document.addEventListener('preserve-language-state', () => {
      this.applyTranslations();
    });

    // Force language preservation on form submissions
    document.addEventListener('submit', () => {
      setTimeout(() => {
        this.applyTranslations();
      }, 100);
    });

    // Preserve language on AJAX requests
    const originalFetch = window.fetch;
    window.fetch = (...args) => {
      return originalFetch(...args).then(response => {
        setTimeout(() => {
          this.debounceApplyTranslations();
        }, 200);
        return response;
      });
    };
  }

  /**
   * Debounced version of applyTranslations to prevent excessive calls
   */
  debounceApplyTranslations() {
    if (this.translationTimeout) {
      clearTimeout(this.translationTimeout);
    }

    this.translationTimeout = setTimeout(() => {
      this.applyTranslations();
    }, 150);
  }

  /**
   * Force apply current language (for emergency use)
   */
  forceApplyCurrentLanguage() {
    if (this.currentLanguage && this.translations[this.currentLanguage]) {
      const langData = this.translations[this.currentLanguage];

      // Force update document attributes
      document.documentElement.lang = langData.meta.code;
      document.documentElement.dir = langData.meta.direction;

      // Force apply all translations
      this.applyTranslations();

      // Force applied language
    }
  }
}

// Initialize the smart language manager when DOM is loaded - remove duplicate initialization
if (document.readyState === 'loading') {
    document.addEventListener("DOMContentLoaded", async () => {
        if (!window.languageManager) {
            window.languageManager = new SmartLanguageManager();
            await window.languageManager.init();
        }

        // Set up global translate function for Alpine.js expressions
        window.$translate = function(key, fallback = null) {
            if (window.languageManager && typeof window.languageManager.getTranslation === 'function') {
                return window.languageManager.getTranslation(key);
            }
            return fallback || key;
        };

        // Also create Alpine magic property if Alpine is available
        if (typeof window.Alpine !== 'undefined' && typeof window.Alpine.magic === 'function') {
            window.Alpine.magic('translate', () => window.$translate);
        }
    });
} else {
    if (!window.languageManager) {
        window.languageManager = new SmartLanguageManager();
        window.languageManager.init();
    }

    // Set up global translate function for Alpine.js expressions
    window.$translate = function(key, fallback = null) {
        if (window.languageManager && typeof window.languageManager.getTranslation === 'function') {
            return window.languageManager.getTranslation(key);
        }
        return fallback || key;
    };

    // Also create Alpine magic property if Alpine is available
    if (typeof window.Alpine !== 'undefined' && typeof window.Alpine.magic === 'function') {
        window.Alpine.magic('translate', () => window.$translate);
    }
}

// Global language preservation system
window.preserveLanguage = function() {
  if (window.languageManager) {
    window.languageManager.forceApplyCurrentLanguage();
  }
};

// Auto-preserve language on dynamic content changes
if (typeof window !== 'undefined') {
  // Intercept common DOM manipulation methods
  const originalSetTimeout = window.setTimeout;
  window.setTimeout = function(callback, delay) {
    return originalSetTimeout.call(this, function() {
      const result = callback.apply(this, arguments);
      if (window.languageManager) {
        window.languageManager.debounceApplyTranslations();
      }
      return result;
    }, delay);
  };

  // Override MutationObserver for additional language preservation
  const OriginalMutationObserver = window.MutationObserver;
  window.MutationObserver = function(callback) {
    return new OriginalMutationObserver(function(mutations, observer) {
      const result = callback.call(this, mutations, observer);
      if (window.languageManager) {
        setTimeout(() => {
          window.languageManager.debounceApplyTranslations();
        }, 100);
      }
      return result;
    });
  };
}
