// ===== Gaming Scrollbar JavaScript =====

class GamingScrollbar {
    constructor() {
        this.progressBar = null;
        this.progressGlow = null;
        this.scrollTimeout = null;
        this.lastScrollTop = 0;
        this.scrollDirection = 'down';

        // Wait for DOM to be ready before initializing
        if (document.readyState === 'loading') {
            document.addEventListener('DOMContentLoaded', () => {
                this.init();
            });
        } else {
            this.init();
        }
    }

    init() {
        this.createProgressBar();
        this.bindScrollEvents();
        this.updateProgress();
        this.addScrollToTop();
        this.addParallaxEffect();
    }

    createProgressBar() {
        // Create progress bar container
        const progressContainer = document.createElement('div');
        progressContainer.className = 'scroll-progress-container';
        progressContainer.innerHTML = `
            <div class="scroll-progress-bar"></div>
            <div class="scroll-progress-glow"></div>
        `;

        document.body.appendChild(progressContainer);
        this.progressBar = progressContainer.querySelector('.scroll-progress-bar');
        this.progressGlow = progressContainer.querySelector('.scroll-progress-glow');
    }

    bindScrollEvents() {
        let ticking = false;

        window.addEventListener('scroll', () => {
            // Determine scroll direction
            const currentScrollTop = window.pageYOffset || document.documentElement.scrollTop;
            this.scrollDirection = currentScrollTop > this.lastScrollTop ? 'down' : 'up';
            this.lastScrollTop = currentScrollTop;

            if (!ticking) {
                requestAnimationFrame(() => {
                    this.updateProgress();
                    this.handleScrollingState();
                    this.updateScrollDirection();
                    ticking = false;
                });
                ticking = true;
            }
        }, { passive: true });

        // Handle smooth scrolling for anchor links
        document.querySelectorAll('a[href^="#"]').forEach(anchor => {
            anchor.addEventListener('click', (e) => {
                const href = anchor.getAttribute('href');

                // Skip if href is just '#' or empty
                if (!href || href === '#' || href.length <= 1) {
                    return;
                }

                e.preventDefault();
                const target = document.querySelector(href);
                if (target) {
                    this.smoothScrollTo(target);
                }
            });
        });

        // Keyboard navigation
        document.addEventListener('keydown', (e) => {
            if (e.key === 'Home') {
                e.preventDefault();
                this.smoothScrollTo(document.body, 1000);
            } else if (e.key === 'End') {
                e.preventDefault();
                this.smoothScrollTo(document.documentElement, 1000);
            }
        });
    }

    updateScrollDirection() {
        document.body.classList.toggle('scrolling-up', this.scrollDirection === 'up');
        document.body.classList.toggle('scrolling-down', this.scrollDirection === 'down');
    }

    updateProgress() {
        const scrollTop = window.pageYOffset || document.documentElement.scrollTop;
        const scrollHeight = document.documentElement.scrollHeight - window.innerHeight;
        const progress = Math.min((scrollTop / scrollHeight) * 100, 100);

        if (this.progressBar) {
            this.progressBar.style.width = `${progress}%`;

            // Add rainbow effect when fully scrolled
            if (progress >= 99) {
                this.progressBar.classList.add('rainbow');
                this.createScrollCompleteEffect();
            } else {
                this.progressBar.classList.remove('rainbow');
            }

            // Dynamic glow based on scroll speed
            if (this.progressGlow) {
                const glowIntensity = Math.min(Math.abs(scrollTop - this.lastScrollTop) / 10, 1);
                this.progressGlow.style.opacity = 0.3 + (glowIntensity * 0.4);
            }
        }
    }

    createScrollCompleteEffect() {
        // Create celebration particles when scroll is complete
        if (!this.hasCreatedCompleteEffect) {
            this.hasCreatedCompleteEffect = true;

            for (let i = 0; i < 5; i++) {
                setTimeout(() => {
                    this.createScrollParticle();
                }, i * 100);
            }

            // Reset after a delay
            setTimeout(() => {
                this.hasCreatedCompleteEffect = false;
            }, 2000);
        }
    }

    createScrollParticle() {
        const particle = document.createElement('div');
        particle.className = 'scroll-complete-particle';
        particle.style.cssText = `
            position: fixed;
            top: 10px;
            left: ${Math.random() * window.innerWidth}px;
            width: 8px;
            height: 8px;
            background: radial-gradient(circle, var(--color-game-neon-green), transparent);
            border-radius: 50%;
            pointer-events: none;
            z-index: 10001;
            animation: scroll-particle-fall 2s ease-out forwards;
        `;

        document.body.appendChild(particle);

        setTimeout(() => {
            if (particle.parentNode) {
                particle.parentNode.removeChild(particle);
            }
        }, 2000);
    }

    handleScrollingState() {
        document.body.classList.add('scrolling');

        // Clear existing timeout
        if (this.scrollTimeout) {
            clearTimeout(this.scrollTimeout);
        }

        // Remove scrolling class after scroll ends
        this.scrollTimeout = setTimeout(() => {
            document.body.classList.remove('scrolling');
        }, 150);
    }

    smoothScrollTo(target, duration = 1000) {
        const targetPosition = target.offsetTop;
        const startPosition = window.pageYOffset;
        const distance = targetPosition - startPosition;
        let startTime = null;

        const easeInOutCubic = (t) => {
            return t < 0.5 ? 4 * t * t * t : (t - 1) * (2 * t - 2) * (2 * t - 2) + 1;
        };

        const animation = (currentTime) => {
            if (startTime === null) startTime = currentTime;
            const timeElapsed = currentTime - startTime;
            const progress = Math.min(timeElapsed / duration, 1);
            const ease = easeInOutCubic(progress);

            window.scrollTo(0, startPosition + (distance * ease));

            if (timeElapsed < duration) {
                requestAnimationFrame(animation);
            }
        };

        requestAnimationFrame(animation);
    }

    addScrollToTop() {
        // Create scroll to top button
        const scrollToTopBtn = document.createElement('div');
        scrollToTopBtn.className = 'scroll-to-top';
        scrollToTopBtn.innerHTML = `
            <svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                <path d="M12 4L8 8H11V16H13V8H16L12 4Z" fill="currentColor"/>
            </svg>
        `;
        document.body.appendChild(scrollToTopBtn);

        // Show/hide button based on scroll position
        window.addEventListener('scroll', () => {
            if (window.pageYOffset > 300) {
                scrollToTopBtn.classList.add('visible');
            } else {
                scrollToTopBtn.classList.remove('visible');
            }
        });

        // Scroll to top on click
        scrollToTopBtn.addEventListener('click', () => {
            this.smoothScrollTo(document.body, 800);
        });
    }

    addParallaxEffect() {
        // Add parallax scrolling effect for elements with .parallax class
        const parallaxElements = document.querySelectorAll('.parallax');

        window.addEventListener('scroll', () => {
            const scrollTop = window.pageYOffset;

            parallaxElements.forEach(element => {
                const speed = element.dataset.speed || 0.5;
                const yPos = -(scrollTop * speed);
                element.style.transform = `translateY(${yPos}px)`;
            });
        });
    }

    // Smooth scroll to specific element by ID
    scrollToElement(elementId) {
        const element = document.getElementById(elementId);
        if (element) {
            this.smoothScrollTo(element);
        }
    }

    // Get current scroll percentage
    getScrollPercentage() {
        const scrollTop = window.pageYOffset || document.documentElement.scrollTop;
        const scrollHeight = document.documentElement.scrollHeight - window.innerHeight;
        return (scrollTop / scrollHeight) * 100;
    }
}

// Initialize scrollbar when DOM is loaded - remove duplicate initialization
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', function() {
        if (!window.gamingScrollbar) {
            window.gamingScrollbar = new GamingScrollbar();
        }
    });
} else {
    if (!window.gamingScrollbar) {
        window.gamingScrollbar = new GamingScrollbar();
    }
}

// Add CSS classes for scroll indicators
document.addEventListener('scroll', function() {
    const scrollPercentage = (window.pageYOffset / (document.documentElement.scrollHeight - window.innerHeight)) * 100;

    // Add classes based on scroll position
    if (scrollPercentage > 10) {
        document.body.classList.add('scrolled');
    } else {
        document.body.classList.remove('scrolled');
    }

    if (scrollPercentage > 90) {
        document.body.classList.add('near-bottom');
    } else {
        document.body.classList.remove('near-bottom');
    }
});
