// ===== Enhanced Responsive Gaming Controls =====

class ResponsiveGamingController {
    constructor() {
        this.breakpoints = {
            mobile: 768,
            tablet: 1024,
            desktop: 1920,
            ultrawide: 2560
        };

        this.currentBreakpoint = this.getCurrentBreakpoint();
        this.touchDevice = this.isTouchDevice();
        this.orientation = this.getOrientation();
        this.isDestroyed = false;
        this.resizeTimeout = null;
        this.orientationTimeout = null;

        this.init();
    }

    init() {
        this.bindEvents();
        this.updateGamingElements();
    }

    getCurrentBreakpoint() {
        const width = window.innerWidth;
        if (width <= this.breakpoints.mobile) return 'mobile';
        if (width <= this.breakpoints.tablet) return 'tablet';
        if (width <= this.breakpoints.desktop) return 'desktop';
        return 'ultrawide';
    }

    isTouchDevice() {
        return (
            'ontouchstart' in window ||
            navigator.maxTouchPoints > 0 ||
            window.matchMedia('(hover: none)').matches ||
            window.matchMedia('(pointer: coarse)').matches
        );
    }

    getOrientation() {
        return window.innerHeight > window.innerWidth ? 'portrait' : 'landscape';
    }

    bindEvents() {
        // Improved debounced resize handler
        const resizeHandler = () => {
            if (this.isDestroyed) return;

            clearTimeout(this.resizeTimeout);
            this.resizeTimeout = setTimeout(() => {
                if (!this.isDestroyed) {
                    this.handleResize();
                }
            }, 150); // Increased debounce time for better performance
        };

        // Orientation change handler
        const orientationHandler = () => {
            if (this.isDestroyed) return;

            clearTimeout(this.orientationTimeout);
            this.orientationTimeout = setTimeout(() => {
                if (!this.isDestroyed) {
                    this.handleOrientationChange();
                }
            }, 400); // Longer delay for orientation changes
        };

        window.addEventListener('resize', resizeHandler, { passive: true });
        window.addEventListener('orientationchange', orientationHandler, { passive: true });

        // Store handlers for cleanup
        this.resizeHandler = resizeHandler;
        this.orientationHandler = orientationHandler;
    }

    handleResize() {
        const newBreakpoint = this.getCurrentBreakpoint();
        const newOrientation = this.getOrientation();

        if (newBreakpoint !== this.currentBreakpoint) {
            this.currentBreakpoint = newBreakpoint;
            this.updateGamingElements();
            this.notifyBreakpointChange(newBreakpoint);
        }

        if (newOrientation !== this.orientation) {
            this.orientation = newOrientation;
            this.handleOrientationChange();
        }
    }

    handleOrientationChange() {
        document.body.classList.remove('portrait', 'landscape');
        document.body.classList.add(this.orientation);

        // Reinitialize gaming elements if needed
        if (this.currentBreakpoint === 'mobile') {
            this.reinitializeForMobile();
        }
    }

    updateGamingElements() {
        document.body.className = document.body.className.replace(
            /\b(mobile|tablet|desktop|ultrawide)\b/g, ''
        );
        document.body.classList.add(this.currentBreakpoint);

        if (this.touchDevice) {
            document.body.classList.add('touch-device');
        } else {
            document.body.classList.remove('touch-device');
        }

        // Update cursor settings
        this.updateCursorSettings();

        // Update scrollbar settings
        this.updateScrollbarSettings();
    }

    updateCursorSettings() {
        if (window.gamingCursor) {
            if (this.touchDevice || this.currentBreakpoint === 'mobile') {
                window.gamingCursor.destroy();
            } else if (!window.gamingCursor.isEnabled) {
                window.gamingCursor.init();
            }
        }
    }

    updateScrollbarSettings() {
        const root = document.documentElement;

        switch(this.currentBreakpoint) {
            case 'mobile':
                root.style.setProperty('--scrollbar-width', '6px');
                root.style.setProperty('--progress-height', '3px');
                break;
            case 'tablet':
                root.style.setProperty('--scrollbar-width', '8px');
                root.style.setProperty('--progress-height', '4px');
                break;
            case 'desktop':
                root.style.setProperty('--scrollbar-width', '14px');
                root.style.setProperty('--progress-height', '6px');
                break;
            case 'ultrawide':
                root.style.setProperty('--scrollbar-width', '18px');
                root.style.setProperty('--progress-height', '8px');
                break;
        }
    }

    reinitializeForMobile() {
        // Remove all cursor elements
        document.querySelectorAll('.cursor, .cursor-trail, .cursor-particle, .cursor-click-ripple').forEach(el => {
            el.remove();
        });

        // Ensure mobile-friendly scrolling
        document.body.style.overflowX = 'hidden';
        document.body.style.webkitOverflowScrolling = 'touch';
    }

    notifyBreakpointChange(newBreakpoint) {
        // Dispatch custom event for other components to listen
        const event = new CustomEvent('gamingBreakpointChange', {
            detail: {
                oldBreakpoint: this.currentBreakpoint,
                newBreakpoint: newBreakpoint,
                touchDevice: this.touchDevice,
                orientation: this.orientation
            }
        });

        document.dispatchEvent(event);
    }

    // Utility methods for other components
    isLowEndDevice() {
        return (
            this.currentBreakpoint === 'mobile' ||
            navigator.hardwareConcurrency <= 2
        );
    }

    shouldReduceAnimations() {
        return (
            this.isLowEndDevice() ||
            window.matchMedia('(prefers-reduced-motion: reduce)').matches
        );
    }

    // Cleanup method
    destroy() {
        this.isDestroyed = true;

        // Clear timeouts
        if (this.resizeTimeout) {
            clearTimeout(this.resizeTimeout);
            this.resizeTimeout = null;
        }

        if (this.orientationTimeout) {
            clearTimeout(this.orientationTimeout);
            this.orientationTimeout = null;
        }

        // Remove event listeners
        if (this.resizeHandler) {
            window.removeEventListener('resize', this.resizeHandler);
            this.resizeHandler = null;
        }

        if (this.orientationHandler) {
            window.removeEventListener('orientationchange', this.orientationHandler);
            this.orientationHandler = null;
        }

        // Responsive Gaming Controller destroyed
    }
}

// Initialize responsive controller
document.addEventListener('DOMContentLoaded', function() {
    window.responsiveGamingController = new ResponsiveGamingController();
});

// Cleanup on page unload
window.addEventListener('beforeunload', () => {
    if (window.responsiveGamingController) {
        window.responsiveGamingController.destroy();
    }
});

// Export for use in other modules
window.ResponsiveGamingController = ResponsiveGamingController;
