/**
 * Player Modal Handler
 * Handles the opening and closing of player detail modals
 */

class PlayerModalHandler {
    constructor() {
        this.isInitialized = false;
        this.isDestroyed = false;
        this.eventListeners = []; // Track event listeners for cleanup
        this.init();
    }

    init() {
        if (this.isInitialized || this.isDestroyed) return;

        // Wait for DOM to be ready
        if (document.readyState === 'loading') {
            document.addEventListener('DOMContentLoaded', () => this.setupEventListeners());
        } else {
            this.setupEventListeners();
        }

        this.isInitialized = true;
    }

    setupEventListeners() {
        if (this.isDestroyed) return;

        // Handle clicks on player avatars and names
        const clickHandler = (e) => this.handlePlayerClick(e);
        document.addEventListener('click', clickHandler);
        this.eventListeners.push({ element: document, event: 'click', handler: clickHandler });

        // Handle escape key to close modal
        const keydownHandler = (e) => {
            if (e.key === 'Escape') {
                this.closeModal();
            }
        };
        document.addEventListener('keydown', keydownHandler);
        this.eventListeners.push({ element: document, event: 'keydown', handler: keydownHandler });

        // Add visual effects
        this.addHoverEffects();

        // Player Modal Handler initialized
    }

    handlePlayerClick(e) {
        // Check if clicked element is a clickable player element
        const clickableElement = e.target.closest('.player-avatar-clickable, .player-small-avatar-clickable, .player-name-clickable');

        if (!clickableElement) return;

        e.preventDefault();
        e.stopPropagation();

        // Find the player container
        const playerContainer = clickableElement.closest('.podium-player, .table-row');

        if (!playerContainer) {
            // Player container not found
            return;
        }

        // Extract player data
        const playerData = this.extractPlayerData(playerContainer, clickableElement);

        if (playerData) {
            this.openModal(playerData);
        }
    }

    extractPlayerData(container, clickedElement) {
        try {
            // Get basic data from container attributes
            const playerId = container.dataset.playerId;
            const rank = container.dataset.rank;
            const points = container.dataset.points;

            // Get avatar source
            const avatarElement = container.querySelector('.player-avatar, .player-small-avatar');
            const avatar = avatarElement ? avatarElement.src : '/assets/images/avatars/default.gif';

            // Get player name
            const nameElement = container.querySelector('.player-name, .player-name-container span');
            const charName = nameElement ? nameElement.textContent.trim() : (playerId || 'Unknown');

            // Check if player is owned
            const isOwned = container.classList.contains('player-owned') ||
                           container.classList.contains('table-row-owned');

            // Extract additional data from stat elements if available
            const killsElement = container.querySelector('.stat-value:first-child, .table-cell:nth-child(5) .text-green-400');
            const deathsElement = container.querySelector('.table-cell:nth-child(6) .text-red-400');
            const jobElement = container.querySelector('.stat-value + .stat-label[data-translate="rankings.job"]');

            return {
                charName: charName,
                name: charName, // fallback
                rank: rank ? parseInt(rank) : null,
                points: points ? parseInt(points) : null,
                kills: killsElement ? parseInt(killsElement.textContent.trim()) : null,
                deaths: deathsElement ? parseInt(deathsElement.textContent.trim()) : null,
                avatar: avatar,
                is_owned: isOwned,
                refObjID: this.extractRefObjFromAvatar(avatar),
                job_type: jobElement ? this.getJobTypeFromName(jobElement.textContent) : null
            };
        } catch (error) {
            // Error extracting player data
            return null;
        }
    }

    // Helper method to extract RefObjID from avatar path
    extractRefObjFromAvatar(avatarPath) {
        try {
            const match = avatarPath.match(/\/(\d+)\.gif$/);
            return match ? parseInt(match[1]) : 1907;
        } catch (error) {
            return 1907; // Default
        }
    }

    // Helper method to convert job name back to job type
    getJobTypeFromName(jobName) {
        const jobTypes = {
            'Trader': 0,
            'Hunter': 1,
            'Thief': 2
        };
        return jobTypes[jobName] || 0;
    }

    openModal(playerData) {
        // Check if Livewire is available
        if (typeof Livewire === 'undefined') {
            // Livewire not found
            return;
        }

        try {
            // Dispatch event to Livewire component
            Livewire.dispatch('player-modal-open', [playerData]);
            // Modal opened for player
        } catch (error) {
            // Error opening modal
        }
    }

    closeModal() {
        // Check if Livewire is available
        if (typeof Livewire === 'undefined') {
            return;
        }

        try {
            // Dispatch close event to the modal component
            Livewire.dispatch('player-modal-close');
            // Modal close event dispatched
        } catch (error) {
            // Error closing modal
        }
    }

    // Add visual feedback for clickable elements
    addHoverEffects() {
        if (this.isDestroyed) return;

        const style = document.createElement('style');
        style.id = 'player-modal-styles';
        style.textContent = `
            .player-avatar-clickable,
            .player-small-avatar-clickable,
            .player-name-clickable {
                cursor: pointer;
                user-select: none;
            }

            .player-avatar-clickable:active,
            .player-small-avatar-clickable:active {
                transform: scale(0.95);
            }

            .player-name-clickable:active {
                transform: translateY(1px);
            }
        `;

        // Remove existing style if present
        const existingStyle = document.getElementById('player-modal-styles');
        if (existingStyle) {
            existingStyle.remove();
        }

        document.head.appendChild(style);
    }

    // Cleanup method
    destroy() {
        this.isDestroyed = true;

        // Remove event listeners
        this.eventListeners.forEach(({ element, event, handler }) => {
            element.removeEventListener(event, handler);
        });
        this.eventListeners = [];

        // Remove added styles
        const style = document.getElementById('player-modal-styles');
        if (style) {
            style.remove();
        }

        this.isInitialized = false;
        // Player Modal Handler destroyed
    }
}

// Prevent multiple initializations
let playerModalHandlerInstance = null;

// Initialize when DOM is ready
document.addEventListener('DOMContentLoaded', () => {
    if (!playerModalHandlerInstance) {
        playerModalHandlerInstance = new PlayerModalHandler();
    }
});

// Also initialize if DOM is already ready
if (document.readyState !== 'loading' && !playerModalHandlerInstance) {
    playerModalHandlerInstance = new PlayerModalHandler();
}

// Cleanup on page unload
window.addEventListener('beforeunload', () => {
    if (playerModalHandlerInstance) {
        playerModalHandlerInstance.destroy();
        playerModalHandlerInstance = null;
    }
});

// Export for potential external use
window.PlayerModalHandler = PlayerModalHandler;
