/**
 * Advanced Notification System
 * Supports: error, warning, notice, alert, success
 */

class NotificationManager {
    constructor() {
        this.container = null;
        this.isDestroyed = false;
        this.activeTimeouts = [];

        // Wait for DOM to be ready before initializing
        if (document.readyState === 'loading') {
            document.addEventListener('DOMContentLoaded', () => {
                this.init();
            });
        } else {
            this.init();
        }
    }

    init() {
        // Create container if not exists
        if (!document.getElementById('notification-container')) {
            this.container = document.createElement('div');
            this.container.id = 'notification-container';
            this.container.className = 'notification-container';
            document.body.appendChild(this.container);
        } else {
            this.container = document.getElementById('notification-container');
        }

        // Listen for Livewire events
        if (window.Livewire) {
            Livewire.on('show-notification', (data) => {
                // Skip if data is empty or invalid
                if (!data || (Array.isArray(data) && data.length === 0)) {
                    // Empty notification data received, skipping
                    return;
                }

                // Handle both array and object formats
                let notificationData = data;
                if (Array.isArray(data) && data.length > 0) {
                    notificationData = data[0];
                }

                // Skip if notification data is invalid
                if (!notificationData || typeof notificationData !== 'object') {
                    // Invalid notification data received, skipping
                    return;
                }

                // Skip if no message provided
                if (!notificationData.message || notificationData.message === '' || notificationData.message === null || notificationData.message === undefined) {
                    // No message in notification data, skipping
                    return;
                }

                this.show(
                    notificationData.type || 'error',
                    notificationData.message,
                    notificationData.duration || 5000,
                    notificationData.params || {}
                );
            });
        }
    }

    show(type, message, duration = 5000, params = {}) {
        // Skip if message is empty or invalid
        if (!message || message === undefined || message === null || message === '' || message.trim() === '') {
            // Cannot show notification with empty message
            return;
        }

        // Prevent duplicate notifications with same content
        const notificationKey = `${type}-${message}-${JSON.stringify(params)}`;
        const now = Date.now();

        if (this.lastNotification && this.lastNotification.key === notificationKey &&
            (now - this.lastNotification.time) < 1000) {
            return; // Skip duplicate notification within 1 second
        }

        this.lastNotification = { key: notificationKey, time: now };

        let finalMessage = message;

        // Define fallback messages in multiple languages
        const fallbacks = {
            'authentication.login_error': {
                'ar': 'اسم المستخدم أو كلمة المرور غير صحيحة',
                'en': 'Invalid username or password'
            },
            'authentication.login_success': {
                'ar': 'تم تسجيل الدخول بنجاح! مرحباً بك [username]',
                'en': 'Login successful! Welcome back [username]'
            },
            'authentication.db_error': {
                'ar': 'خطأ في قاعدة البيانات. يرجى المحاولة مرة أخرى لاحقاً',
                'en': 'Database error. Please try again later'
            },
            'notification_error': {
                'ar': 'حدث خطأ غير متوقع',
                'en': 'An unexpected error occurred'
            }
        };

        // Get current language - default to English
        let currentLang = 'en';
        if (document.documentElement.lang && document.documentElement.lang !== 'en') {
            currentLang = document.documentElement.lang;
        }

        // Check if message is a translation key with "trans:" prefix
        const isTransPrefix = typeof message === 'string' && message.startsWith('trans:');

        // Check if message looks like a translation key (contains dots and no spaces)
        const isTranslationKey = typeof message === 'string' && (isTransPrefix || (message.includes('.') && !message.includes(' ')));

        if (isTranslationKey) {
            let translationKey = message;

            // Remove "trans:" prefix if present
            if (isTransPrefix) {
                translationKey = message.substring(6); // Remove "trans:" prefix
            }

            // Try to translate using language manager FIRST
            if (window.languageManager && typeof window.languageManager.getTranslation === 'function') {
                const translated = window.languageManager.getTranslation(translationKey);

                // Check if translation was successful (not the same as the key)
                if (translated && translated !== translationKey && translated !== undefined) {
                    finalMessage = translated;
                } else {
                    // Translation failed, try fallback
                    if (fallbacks[translationKey]) {
                        finalMessage = fallbacks[translationKey][currentLang] || fallbacks[translationKey]['en'] || translationKey;
                    } else {
                        // No fallback available, use generic error
                        finalMessage = fallbacks['notification_error'][currentLang] || fallbacks['notification_error']['en'];
                    }
                }
            } else {
                // No language manager available, use fallbacks directly
                if (fallbacks[translationKey]) {
                    finalMessage = fallbacks[translationKey][currentLang] || fallbacks[translationKey]['en'] || translationKey;
                } else {
                    finalMessage = fallbacks['notification_error'][currentLang] || fallbacks['notification_error']['en'];
                }
            }
        } else {
            // Message is not a translation key, use as is if it contains spaces
            if (typeof message === 'string' && message.includes(' ')) {
                finalMessage = message;
            } else {
                // Fallback for unknown cases
                finalMessage = fallbacks['notification_error'][currentLang] || fallbacks['notification_error']['en'];
            }
        }

        // Handle parameter replacement - safe replacement to avoid duplication
        if (params && typeof params === 'object') {
            Object.keys(params).forEach(key => {
                const placeholder = `[${key}]`;
                let value = params[key];

                // Ensure value is a string and doesn't contain the placeholder itself
                if (typeof value !== 'string') {
                    value = String(value || '');
                }

                // Escape special regex characters in placeholder
                const escapedPlaceholder = placeholder.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');

                // Replace all occurrences safely by using a temporary placeholder
                const tempPlaceholder = `__TEMP_${key}_${Math.random().toString(36).substr(2, 9)}__`;
                finalMessage = finalMessage.replace(new RegExp(escapedPlaceholder, 'g'), tempPlaceholder);
                finalMessage = finalMessage.replace(new RegExp(tempPlaceholder.replace(/[.*+?^${}()|[\]\\]/g, '\\$&'), 'g'), value);
            });
        }

        // Create notification element
        const notification = document.createElement('div');
        notification.className = `notification notification-${type}`;

        const icon = this.getIcon(type);
        const title = this.getTitle(type);

        notification.innerHTML = `
            <div class="notification-content">
                <div class="notification-icon">
                    <i class="${icon}"></i>
                </div>
                <div class="notification-text">
                    <div class="notification-title">${title}</div>
                    <div class="notification-message">${finalMessage}</div>
                </div>
                <button class="notification-close" onclick="this.parentElement.parentElement.remove()">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            <div class="notification-progress">
                <div class="notification-progress-bar" style="animation-duration: ${duration}ms"></div>
            </div>
        `;

        // Add to container
        this.container.appendChild(notification);

        // Auto remove after duration
        const timeoutId = setTimeout(() => {
            if (notification.parentElement && !this.isDestroyed) {
                notification.classList.add('notification-hide');
                const hideTimeoutId = setTimeout(() => {
                    if (notification.parentElement && !this.isDestroyed) {
                        notification.remove();
                    }
                }, 300);
                this.activeTimeouts.push(hideTimeoutId);
            }
        }, duration);

        this.activeTimeouts.push(timeoutId);

        // Add show animation
        const showTimeoutId = setTimeout(() => {
            if (!this.isDestroyed) {
                notification.classList.add('notification-show');
            }
        }, 10);

        this.activeTimeouts.push(showTimeoutId);
    }

    getIcon(type) {
        const icons = {
            error: 'fas fa-exclamation-triangle',
            warning: 'fas fa-exclamation-circle',
            notice: 'fas fa-info-circle',
            alert: 'fas fa-bell',
            success: 'fas fa-check-circle'
        };
        return icons[type] || 'fas fa-info-circle';
    }

    getTitle(type) {
        // Try to get translated title first
        if (window.languageManager && typeof window.languageManager.getTranslation === 'function') {
            const translationKey = `notifications.${type}`;
            const translated = window.languageManager.getTranslation(translationKey);
            if (translated && translated !== translationKey) {
                return translated;
            }
        }

        // Fallback titles based on current language
        let currentLang = 'ar';
        if (document.documentElement.lang && document.documentElement.lang !== 'ar') {
            currentLang = document.documentElement.lang;
        }

        const titles = {
            error: { 'ar': 'خطأ', 'en': 'Error' },
            warning: { 'ar': 'تحذير', 'en': 'Warning' },
            notice: { 'ar': 'إخطار', 'en': 'Notice' },
            alert: { 'ar': 'تنبيه', 'en': 'Alert' },
            success: { 'ar': 'نجح', 'en': 'Success' }
        };

        if (titles[type]) {
            return titles[type][currentLang] || titles[type]['ar'];
        }

        return currentLang === 'ar' ? 'إشعار' : 'Notification';
    }

    // Method to manually show notification
    showManual(type, message, duration = 5000) {
        this.show(type, message, duration);
    }

    // Method to clear all notifications
    clearAll() {
        if (this.container) {
            this.container.innerHTML = '';
        }

        // Clear active timeouts
        this.activeTimeouts.forEach(timeoutId => {
            clearTimeout(timeoutId);
        });
        this.activeTimeouts = [];
    }

    // Cleanup method
    destroy() {
        this.isDestroyed = true;

        // Clear all timeouts
        this.activeTimeouts.forEach(timeoutId => {
            clearTimeout(timeoutId);
        });
        this.activeTimeouts = [];

        // Remove container
        if (this.container && this.container.parentNode) {
            this.container.parentNode.removeChild(this.container);
        }
        this.container = null;

        // Notification Manager destroyed
    }
}

// Prevent duplicate initialization - single instance pattern
let notificationManagerInitialized = false;

// Initialize when DOM is ready, but wait for language manager
function initializeNotificationManager() {
    if (notificationManagerInitialized || window.notificationManager) {
        return;
    }

    notificationManagerInitialized = true;

    // Wait a bit for language manager to initialize
    setTimeout(() => {
        if (!window.notificationManager) {
            window.notificationManager = new NotificationManager();
        }
    }, 100);
}

if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', initializeNotificationManager);
} else {
    initializeNotificationManager();
}

// Cleanup on page unload
window.addEventListener('beforeunload', () => {
    if (window.notificationManager) {
        window.notificationManager.destroy();
    }
});

// Global function for easy access
window.showNotification = (type, message, duration) => {
    if (window.notificationManager) {
        window.notificationManager.show(type, message, duration);
    }
};

// Helper function for translation-based notifications
window.showTransNotification = (type, translationKey, duration, params) => {
    if (window.notificationManager) {
        // Add "trans:" prefix to ensure translation
        const message = translationKey.startsWith('trans:') ? translationKey : `trans:${translationKey}`;
        window.notificationManager.show(type, message, duration, params);
    }
};
