// Inventory Tooltips Alpine.js Component
// This registers the component with Alpine.js

Alpine.data('inventoryTooltip', (charId, processedItems) => ({
    // Data properties
    charId: charId,
    processedItems: processedItems,
    tooltipVisible: false,
    tooltipContent: '',
    tooltipX: 0,
    tooltipY: 0,
    currentSlot: null,
    isLoading: false,
    isHoveringTooltip: false,
    hideTooltipTimeout: null,

    // Initialize method
    init() {
        // Listen for tooltip hover events
        this.$watch('isHoveringTooltip', (value) => {
            if (value) {
                // Clear any pending hide timeout
                if (this.hideTooltipTimeout) {
                    clearTimeout(this.hideTooltipTimeout);
                    this.hideTooltipTimeout = null;
                }
            }
        });

        // Listen for custom tooltip hover events
        this.$el.addEventListener('tooltip-hover', (event) => {
            // Tooltip hover event received
            this.isHoveringTooltip = event.detail;

            if (event.detail) {
                // Mouse entered tooltip
                this.onTooltipMouseEnter();
            } else {
                // Mouse left tooltip
                this.onTooltipMouseLeave();
            }
        });
    },

    // Methods
    async showTooltip(event, slot, slotName) {
        // Skip if no item in this slot
        if (!this.processedItems[slot]) {
            return;
        }

        // Debug: log the item data
        // Debug - Item data for slot

        const refObjId = this.processedItems[slot].ref_obj_id;
        if (!refObjId) {
            return;
        }

        // If already showing this tooltip, just update position
        if (this.currentSlot === slot && this.tooltipVisible) {
            this.updateTooltipPosition(event);
            return;
        }

        this.currentSlot = slot;

        // Force tooltip to be visible first
        this.tooltipVisible = true;
        this.updateTooltipPosition(event);

        // Show loading state
        this.tooltipContent = this.getLoadingHTML(slotName);
        this.tooltipVisible = true;
        this.isLoading = true;

        try {
            // Fetching enhanced item details from InventoryService
            const response = await fetch(`/api/inventory/tooltip/${refObjId}`);

            // API Response received

            if (response.ok) {
                const contentType = response.headers.get('content-type');

                if (contentType && contentType.includes('application/json')) {
                    // Handle JSON response (for debugging)
                    const jsonData = await response.json();
                    // JSON Response received

                    if (this.currentSlot === slot) {
                        if (jsonData.success) {
                            // Create HTML from JSON data
                            this.tooltipContent = this.createTooltipFromData(jsonData.item);
                        } else {
                            this.tooltipContent = this.getErrorHTML(slotName, jsonData.message || 'Unknown error');
                        }
                    }
                } else {
                    // Handle HTML response
                    const htmlContent = await response.text();
                    // Successfully received HTML content

                    if (this.currentSlot === slot) {
                        if (htmlContent.trim().length > 0) {
                            // Check if HTML contains the new tooltip structure
                            if (htmlContent.includes('item-tooltip')) {
                                this.tooltipContent = htmlContent;
                            } else {
                                // Wrap legacy HTML in new tooltip structure
                                this.tooltipContent = this.wrapLegacyHTML(htmlContent, slotName);
                            }
                        } else {
                            // API returned empty content
                            this.tooltipContent = this.getErrorHTML(slotName, 'Empty response from server');
                        }
                    }
                }
            } else {
                // API Error
                const errorText = await response.text();
                // Error response

                if (this.currentSlot === slot) {
                    this.tooltipContent = this.getErrorHTML(slotName, `Server error: ${response.status}`);
                }
            }
        } catch (error) {
            // Network/Fetch Error
            if (this.currentSlot === slot) {
                this.tooltipContent = this.getErrorHTML(slotName, 'Network connection error');
            }
        } finally {
            this.isLoading = false;
        }
    },

    createTooltipFromData(item) {
        if (!item) {
            return this.getErrorHTML('Normal', 'No item data provided');
        }

        // Parse SML2 content if available
        let parsedDescription = item.localized_description || item.description || '';
        if (parsedDescription) {
            parsedDescription = this.parseSml2Content(parsedDescription);
        }

        return `
            <div class="item-tooltip inventory-tooltip enhanced-tooltip"
                 x-data="{ tooltipHover: false }"
                 @mouseenter="tooltipHover = true; $dispatch('tooltip-hover', true)"
                 @mouseleave="tooltipHover = false; $dispatch('tooltip-hover', false)"
                 @wheel.stop="">
                <!-- Enhanced Header with InventoryService Data -->
                <div class="tooltip-header-enhanced ${this.getGlowClass(item.glow_level)}">
                    <div class="icon-and-name">
                        <!-- Item Icon with Grade -->
                        <div class="tooltip-icon ${item.grade || 'normal'}">
                            <img src="${item.icon_path || '/assets/images/ui/placeholder-item.svg'}"
                                 alt="${item.name || 'Unknown Item'}"
                                 class="item-icon-tooltip ${item.grade || 'normal'}"
                                 onerror="this.src='/assets/images/ui/placeholder-item.svg';">
                        </div>

                        <!-- Item Name and Type -->
                        <div class="item-name-section">
                            <div class="item-details">
                                <span class="rarity-badge ${item.grade || 'normal'}">
                                    ${item.rarity || 'Normal'}
                                </span>
                                ${item.sort_of_item ? `<span class="sort-of-item-badge">${item.sort_of_item}</span>` : ''}
                                ${item.degree ? `<span class="degree-badge">D${item.degree}</span>` : ''}
                                ${item.race ? `<span class="race-badge">${item.race}</span>` : ''}
                                <br><hr class="tooltip-separator">
                                ${item.opt_level > 0 && item.glow_level > 0 ? `<span class="opt-level">Main +${item.opt_level-item.glow_level}</span>` : ''}
                                ${item.glow_level > 0 ? `<span class="advanced-badge ${this.getGlowClass(item.glow_level)}">Advanced +${item.glow_level}</span>` : ''}
                                ${item.opt_level > 0 ? `<span class="opt-level-badge">Total +${item.opt_level}</span>` : ''}
                            </div>
                        </div>
                    </div>

                    <!-- SOX Information -->
                    ${item.sox ? `
                        <div class="sox-info">
                            <div class="sox-badge ${item.grade || 'normal'}">
                                <i class="fas fa-gem"></i>
                                <span>${item.sox}</span>
                            </div>
                        </div>
                    ` : ''}

                    <hr class="tooltip-separator">
                </div>

                <!-- Enhanced Body with InventoryService Stats -->
                <div class="tooltip-body-enhanced" @wheel.stop="">
                    <!-- Item Description -->
                    ${parsedDescription ? `
                        <div class="item-description sml2-content">
                            ${parsedDescription}
                        </div>
                    ` : ''}




                    <!-- White Stats (Enhancement Stats) -->
                    ${this.renderWhiteStats(item.white_stats && item.white_stats.length > 0 ? item.white_stats : this.getTestWhiteStats(), item)}



                    <!-- Blue Stats (Magic Options) -->
                    ${this.renderBlueStats(item.blue_stats && item.blue_stats.length > 0 ? item.blue_stats : this.getTestBlueStats())}

                    <!-- Requirements -->
                    ${this.renderEnhancedRequirements(item)}

                    <!-- Pet Information -->
                    ${item.is_pet && item.pet_info ? this.renderPetInfo(item.pet_info) : ''}

                    <!-- Trading Properties -->
                    <div class="trading-properties">
                        <h5 class="section-title">
                            <i class="fas fa-exchange-alt"></i>
                            Trading Properties
                        </h5>
                        <div class="properties-grid">
                            <div class="property ${item.can_trade ? 'allowed' : 'forbidden'}">
                                <i class="fas ${item.can_trade ? 'fa-check' : 'fa-times'}"></i>
                                <span>${item.can_trade ? 'Tradeable' : 'Not Tradeable'}</span>
                            </div>
                            <div class="property ${item.can_sell ? 'allowed' : 'forbidden'}">
                                <i class="fas ${item.can_sell ? 'fa-check' : 'fa-times'}"></i>
                                <span>${item.can_sell ? 'Sellable' : 'Not Sellable'}</span>
                            </div>
                            <div class="property ${item.can_drop ? 'allowed' : 'forbidden'}">
                                <i class="fas ${item.can_drop ? 'fa-check' : 'fa-times'}"></i>
                                <span>${item.can_drop ? 'Droppable' : 'Can\'t Drop'}</span>
                            </div>
                        </div>
                    </div>

                    <!-- Price Information -->
                    ${item.price > 0 ? `
                        <div class="price-section">
                            <h5 class="section-title">
                                <i class="fas fa-coins"></i>
                                NPC Price
                            </h5>
                            <div class="price-value">
                                <span class="gold-amount">${this.formatNumber(item.price)}</span>
                                <span class="currency">Gold</span>
                            </div>
                        </div>
                    ` : ''}

                    <!-- Debug Info -->
                    ${item.raw_data ? `
                        <div class="debug-info">
                            <details>
                                <summary>Debug Information</summary>
                                <small class="debug-details">
                                    ID: ${item.id || 'N/A'} |
                                    Code: ${item.code_name || 'N/A'}<br>
                                    Type: ${item.raw_data.TypeID2}/${item.raw_data.TypeID3}/${item.raw_data.TypeID4}<br>
                                    Class: ${item.raw_data.ItemClass} |
                                    Rarity: ${item.raw_data.Rarity}
                                </small>
                            </details>
                        </div>
                    ` : ''}


                </div>
            </div>
        `;
    },

    hideTooltip() {
        // Clear any existing timeout
        if (this.hideTooltipTimeout) {
            clearTimeout(this.hideTooltipTimeout);
        }

        // Set a delay to allow user to move mouse to tooltip
        this.hideTooltipTimeout = setTimeout(() => {
            if (!this.isHoveringTooltip) {
                this.tooltipVisible = false;
                this.currentSlot = null;
                this.tooltipContent = '';
                this.isLoading = false;
                this.hideTooltipTimeout = null;
            }
        }, 150); // Increased delay to 150ms
    },

    // Enhanced methods to handle tooltip hover
    onTooltipMouseEnter() {
        this.isHoveringTooltip = true;
        // Clear any pending hide timeout
        if (this.hideTooltipTimeout) {
            clearTimeout(this.hideTooltipTimeout);
            this.hideTooltipTimeout = null;
        }

        // Add wheel event listener to prevent scroll propagation
        this.addTooltipScrollHandler();
    },

    onTooltipMouseLeave() {
        this.isHoveringTooltip = false;
        this.hideTooltip();

        // Remove wheel event listener
        this.removeTooltipScrollHandler();
    },

    // Add scroll handling for tooltip
    addTooltipScrollHandler() {
        const tooltipBody = document.querySelector('.tooltip-body-enhanced');
        if (tooltipBody) {
            tooltipBody.addEventListener('wheel', this.handleTooltipWheel.bind(this), { passive: false });
        }
    },

    // Remove scroll handling for tooltip
    removeTooltipScrollHandler() {
        const tooltipBody = document.querySelector('.tooltip-body-enhanced');
        if (tooltipBody) {
            tooltipBody.removeEventListener('wheel', this.handleTooltipWheel.bind(this));
        }
    },

    // Handle wheel events inside tooltip
    handleTooltipWheel(event) {
        const tooltipBody = event.currentTarget;
        const scrollTop = tooltipBody.scrollTop;
        const scrollHeight = tooltipBody.scrollHeight;
        const clientHeight = tooltipBody.clientHeight;

        // Wheel Event processing

        // Always prevent page scroll when inside tooltip
        event.stopPropagation();
        event.preventDefault();

        // Manually handle scrolling
        const scrollAmount = event.deltaY > 0 ? 50 : -50;
        tooltipBody.scrollTop += scrollAmount;
    },

    // Method to handle slot mouse leave
    onSlotMouseLeave() {
        // Only hide if we're not hovering over the tooltip
        if (!this.isHoveringTooltip) {
            this.hideTooltip();
        }
    },

    updateTooltipPosition(event) {
        if (!this.tooltipVisible) return;

        const padding = 20;
        const tooltipWidth = 350;
        const tooltipHeight = 450;

        // Get viewport dimensions
        const viewportWidth = window.innerWidth || document.documentElement.clientWidth;
        const viewportHeight = window.innerHeight || document.documentElement.clientHeight;

        // Start with cursor position
        let x = event.clientX;
        let y = event.clientY;

        // Determine best position (right/left of cursor)
        if (x + tooltipWidth + padding > viewportWidth) {
            // Show to the left of cursor
            x = event.clientX - tooltipWidth - padding;
        } else {
            // Show to the right of cursor
            x = event.clientX + padding;
        }

        // Determine vertical position (below/above cursor)
        if (y + tooltipHeight + padding > viewportHeight) {
            // Show above cursor
            y = event.clientY - tooltipHeight - padding;
        } else {
            // Show below cursor
            y = event.clientY + padding;
        }

        // Ensure tooltip stays within viewport bounds
        x = Math.max(padding, Math.min(x, viewportWidth - tooltipWidth - padding));
        y = Math.max(padding, Math.min(y, viewportHeight - tooltipHeight - padding));

        // Set the final position
        this.tooltipX = x;
        this.tooltipY = y;
    },

    // Helper function to parse SML2 content
    parseSml2Content(content) {
        if (!content) return '';

        // Remove <sml2> tags
        content = content.replace(/<sml2>(.*?)<\/sml2>/gs, '$1');

        // Convert font color tags to HTML spans
        content = content.replace(/<font color="([^"]+)">(.*?)<\/font>/gi, (match, color, text) => {
            const colorValues = color.split(',');
            if (colorValues.length >= 3) {
                const r = parseInt(colorValues[0]) || 0;
                const g = parseInt(colorValues[1]) || 0;
                const b = parseInt(colorValues[2]) || 0;
                const a = colorValues[3] ? parseFloat(colorValues[3]) / 255 : 1;

                const rgbaColor = `rgba(${r}, ${g}, ${b}, ${a})`;
                return `<span style="color: ${rgbaColor}">${text}</span>`;
            }
            return text;
        });

        // Convert basic HTML tags
        content = content.replace(/<strong>/g, '<strong class="sml2-strong">');
        content = content.replace(/<br\s*\/?>/gi, '<br class="sml2-br">');

        return content;
    },

    // Helper function to render blue stats (magic options)
    renderBlueStats(blueStats) {
        if (!blueStats || blueStats.length === 0) return '';

        // Rendering BlueStats

        const statsHtml = blueStats.map(stat => {
            const isNegative = stat.is_negative === true || stat.color === '#ff2f51';
            const categoryClass = `blue-stat-${stat.category || 'enhancement'}`;
            const negativeClass = isNegative ? 'negative-stat' : 'positive-stat';

            return `
                <div class="blue-stat-item ${categoryClass} ${negativeClass}" data-category="${stat.category || 'enhancement'}">
                    <div class="stat-icon">
                        ${stat.icon ? `<i class="${stat.icon}"></i>` : '<i class="fas fa-star"></i>'}
                    </div>
                    <div class="stat-content">
                        <span class="stat-name" style="color: ${stat.color || '#50cecd'}">
                            ${stat.display_name || stat.name}
                        </span>
                        ${stat.description ? `<div class="stat-description">${stat.description}</div>` : ''}
                    </div>
                    <div class="stat-meta">
                        <span class="stat-sort-key">#${stat.sort_key || stat.id || 0}</span>
                    </div>
                </div>
            `;
        }).join('');

        // Group stats by category for better organization
        const categories = [...new Set(blueStats.map(stat => stat.category || 'enhancement'))];
        const categorizedStats = categories.map(category => {
            const categoryStats = blueStats.filter(stat => (stat.category || 'enhancement') === category);
            const categoryIcon = this.getBlueStatCategoryIcon(category);
            const categoryName = this.getBlueStatCategoryName(category);

            return `
                <div class="blue-stats-category" data-category="${category}">
                    <div class="category-header">
                        <i class="${categoryIcon}"></i>
                        <span class="category-name">${categoryName}</span>
                        <span class="category-count">(${categoryStats.length})</span>
                    </div>
                    <div class="category-stats">
                        ${categoryStats.map(stat => {
                            const isNegative = stat.is_negative === true || stat.color === '#ff2f51';
                            const negativeClass = isNegative ? 'negative-stat' : 'positive-stat';

                            return `
                                <div class="blue-stat-item ${negativeClass}">
                                    <div class="stat-icon">
                                        ${stat.icon ? `<i class="${stat.icon}"></i>` : '<i class="fas fa-star"></i>'}
                                    </div>
                                    <div class="stat-content">
                                        <span class="stat-name" style="color: ${stat.color || '#50cecd'}">
                                            ${stat.display_name || stat.name}
                                        </span>
                                        ${stat.description ? `<div class="stat-description">${stat.description}</div>` : ''}
                                    </div>
                                </div>
                            `;
                        }).join('')}
                    </div>
                </div>
            `;
        }).join('');

        return `
            <div class="blue-stats-section enhanced-blue-stats">
                <h5 class="section-title blue-title">
                    <i class="fas fa-magic"></i>
                    <span class="title-text">Blue Stats (Magic Options)</span>
                    <div class="stats-summary">
                        <span class="stats-count">${blueStats.length} Options</span>
                    </div>
                </h5>
                <div class="blue-stats-container">
                    ${categorizedStats}
                </div>
                <div class="blue-stats-footer">
                    <div class="stats-legend">
                        <span class="legend-item positive">
                            <i class="fas fa-arrow-up"></i> Enhancement
                        </span>
                        <span class="legend-item negative">
                            <i class="fas fa-arrow-down"></i> Penalty
                        </span>
                    </div>
                </div>
            </div>
        `;
    },

    // Helper function to render white stats (enhancement stats)
    renderWhiteStats(whiteStats, item) {
        if (!whiteStats || whiteStats.length === 0) return '';

        // Rendering WhiteStats

        // Determine item type for proper ordering
        const itemType = this.getItemTypeForStats(item);

        // Parse and organize stats based on item type
        const organizedStats = this.organizeWhiteStatsByType(whiteStats, itemType);

        // Filter out zero values and unwanted stats
        const filteredStats = organizedStats.filter(stat => this.shouldShowWhiteStat(stat));

        if (filteredStats.length === 0) return '';

        // Generate ordered stats HTML
        const statsHtml = filteredStats.map(stat => {
            const statIcon = this.getWhiteStatIcon(stat.type);
            return `
                <div class="white-stat-item" data-type="${stat.type}">
                    <div class="stat-icon">
                        <i class="${statIcon}"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-text">${stat.display_text}</div>
                    </div>
                </div>
            `;
        }).join('');

        return `
            <div class="white-stats-section enhanced-white-stats">
                <h5 class="section-title white-title">
                    <i class="fas fa-chart-line"></i>
                    <span class="title-text">White Stats</span>
                    <div class="stats-summary">
                        <span class="stats-count">${filteredStats.length} Stats</span>
                    </div>
                </h5>
                <div class="white-stats-container simple-list">
                    ${statsHtml}
                </div>
            </div>
        `;
    },

    // Helper function to determine item type for stats organization
    getItemTypeForStats(item) {
        const typeId3 = item?.raw_data?.TypeID3 || item?.type_ids?.TypeID3 || 0;

        switch (typeId3) {
            case 6: return 'weapon';      // Weapons
            case 4: return 'shield';      // Shields
            case 2: return 'shield';      // Also shields
            case 1: return 'armor';       // Armor/Clothing
            case 3: return 'accessory';   // Accessories
            case 5: return 'accessory';   // Also accessories
            default: return 'other';
        }
    },

    // Helper function to organize stats based on item type
    organizeWhiteStatsByType(whiteStats, itemType) {
        const organized = [];
        const statsMap = new Map();

        // Parse all stats and categorize them
        whiteStats.forEach(stat => {
            const text = (stat.raw_text || stat.display_text || '').toLowerCase();
            const parsedStat = this.parseWhiteStatText(text, stat);
            if (parsedStat) {
                statsMap.set(parsedStat.type, parsedStat);
            }
        });

        // Define order based on item type
        let order = [];
        switch (itemType) {
            case 'weapon':
                order = ['phy_attack', 'mag_attack', 'durability', 'attack_distance', 'attack_rate', 'critical', 'phy_reinforce', 'mag_reinforce'];
                break;
            case 'shield':
                order = ['phy_defense', 'mag_defense', 'durability', 'blocking', 'phy_reinforce', 'mag_reinforce'];
                break;
            case 'armor':
                order = ['phy_defense', 'mag_defense', 'durability', 'parry_rate', 'phy_reinforce', 'mag_reinforce'];
                break;
            case 'accessory':
                order = ['phy_absorption', 'mag_absorption'];
                break;
            default:
                order = ['durability', 'phy_attack', 'mag_attack', 'phy_defense', 'mag_defense'];
        }

        // Add stats in the specified order
        order.forEach(type => {
            if (statsMap.has(type)) {
                organized.push(statsMap.get(type));
            }
        });

        // Add any remaining stats that weren't in the order
        statsMap.forEach((stat, type) => {
            if (!order.includes(type)) {
                organized.push(stat);
            }
        });

        return organized;
    },

    // Helper function to parse white stat text and determine type
    parseWhiteStatText(text, originalStat) {
        const patterns = {
            'phy_attack': /phy.*atk|physical.*attack/i,
            'mag_attack': /mag.*atk|magical.*attack/i,
            'phy_defense': /phy.*def|physical.*def/i,
            'mag_defense': /mag.*def|magical.*def/i,
            'durability': /durability/i,
            'attack_distance': /attack.*distance|range/i,
            'attack_rate': /attack.*rate|hit.*rate/i,
            'critical': /critical/i,
            'blocking': /block/i,
            'parry_rate': /parry/i,
            'phy_absorption': /phy.*abs|physical.*abs/i,
            'mag_absorption': /mag.*abs|magical.*abs/i,
            'phy_reinforce': /phy.*rein|physical.*rein/i,
            'mag_reinforce': /mag.*rein|magical.*rein/i
        };

        for (const [type, pattern] of Object.entries(patterns)) {
            if (pattern.test(text)) {
                return {
                    type: type,
                    display_text: originalStat.display_text || originalStat.raw_text,
                    value: this.extractStatValue(text),
                    original: originalStat
                };
            }
        }

        // Fallback for unrecognized stats
        return {
            type: 'other',
            display_text: originalStat.display_text || originalStat.raw_text,
            value: this.extractStatValue(text),
            original: originalStat
        };
    },

    // Helper function to extract numeric value from stat text
    extractStatValue(text) {
        const match = text.match(/(\d+)/);
        return match ? parseInt(match[1]) : 0;
    },

    // Helper function to determine if a stat should be shown
    shouldShowWhiteStat(stat) {
        // Always show durability
        if (stat.type === 'durability') return true;

        // Hide zero values for attack/defense stats
        const hideZeroTypes = ['phy_attack', 'mag_attack', 'phy_defense', 'mag_defense', 'phy_reinforce', 'mag_reinforce'];
        if (hideZeroTypes.includes(stat.type) && stat.value === 0) {
            return false;
        }

        return true;
    },

    // Helper function to render enhanced requirements
    renderEnhancedRequirements(item) {
        const requirements = [];

        if (item.req_level > 0) {
            requirements.push({
                type: 'Level',
                value: item.req_level,
                icon: 'fa-level-up-alt'
            });
        }

        if (item.requirements && Array.isArray(item.requirements)) {
            item.requirements.forEach(req => {
                requirements.push({
                    type: req.name,
                    value: req.value,
                    icon: this.getRequirementIcon(req.name)
                });
            });
        }

        if (requirements.length === 0) return '';

        const reqList = requirements.map(req => `
            <div class="requirement-item">
                <i class="fas ${req.icon}"></i>
                <span class="req-type">${req.type}:</span>
                <span class="req-value">${req.value}</span>
            </div>
        `).join('');

        return `
            <div class="requirements-section">
                <h5 class="section-title">
                    <i class="fas fa-exclamation-triangle"></i>
                    Requirements
                </h5>
                <div class="requirements-grid">
                    ${reqList}
                </div>
            </div>
        `;
    },

    // Helper function to render pet information
    renderPetInfo(petInfo) {
        if (!petInfo) return '';

        return `
            <div class="pet-info-section">
                <h5 class="section-title">
                    <i class="fas fa-paw"></i>
                    Pet Information
                </h5>
                <div class="pet-details">
                    ${petInfo.PetName ? `<div class="pet-name">Name: ${petInfo.PetName}</div>` : ''}
                    ${petInfo.PetLevel ? `<div class="pet-level">Level: ${petInfo.PetLevel}</div>` : ''}
                    ${petInfo.PetType ? `<div class="pet-type">Type: ${petInfo.PetType}</div>` : ''}
                </div>
            </div>
        `;
    },

    // Helper function to get requirement icon
    getRequirementIcon(reqName) {
        const iconMap = {
            'Level': 'fa-level-up-alt',
            'Strength': 'fa-fist-raised',
            'Intelligence': 'fa-brain',
            'Skill Points': 'fa-star',
            'Job Type': 'fa-briefcase'
        };
        return iconMap[reqName] || 'fa-info-circle';
    },

    // Helper function to render requirements (legacy)
    renderRequirements(requirements) {
        if (!requirements || !Array.isArray(requirements)) return '';

        const reqTypes = {
            1: 'Level',
            2: 'Strength',
            3: 'Intelligence',
            4: 'Skill Points',
            5: 'Job Type'
        };

        const reqList = requirements.map(req => `
            <li class="requirement-item">
                <span class="req-type">${reqTypes[req.type] || 'Type ' + req.type}:</span>
                <span class="req-value">${req.level}</span>
            </li>
        `).join('');

        return `
            <div class="requirements-section">
                <h5 class="section-title">Requirements:</h5>
                <ul class="requirements-list">
                    ${reqList}
                </ul>
            </div>
        `;
    },

    // Helper function to format numbers
    formatNumber(num) {
        return new Intl.NumberFormat().format(num);
    },

    // Helper function to determine grade from CodeName128
    getGradeFromCodeName(codeName) {
        if (!codeName || !codeName.includes('RARE')) {
            return 'normal';
        }

        if (codeName.includes('A_RARE')) {
            return 'rare';  // Seal of Star
        } else if (codeName.includes('B_RARE')) {
            return 'epic';  // Seal of Moon
        } else if (/[C-Z]_RARE/.test(codeName)) {
            return 'legendary';  // Seal of Sun (C_RARE, D_RARE, E_RARE, F_RARE, G_RARE, etc.)
        } else {
            // Generic RARE without letter prefix
            return 'rare';
        }
    },

    // Helper function to get rarity display name
    getRarityDisplayName(codeName) {
        if (!codeName || !codeName.includes('RARE')) {
            return 'Normal';
        }

        if (codeName.includes('A_RARE')) {
            return 'Seal of Star';
        } else if (codeName.includes('B_RARE')) {
            return 'Seal of Moon';
        } else if (/[C-Z]_RARE/.test(codeName)) {
            return 'Seal of Sun';
        } else {
            return 'Rare';
        }
    },

    // Helper function to wrap legacy HTML in new tooltip structure
    wrapLegacyHTML(htmlContent, slotName) {
        const item = this.processedItems[this.currentSlot];
        return `
            <div class="item-tooltip inventory-tooltip legacy-tooltip">
                <div class="tooltip-header-new">
                    <div class="icon-and-name">
                        <div class="tooltip-icon">
                            <img src="${item.image}"
                                 alt="${item.name}"
                                 class="item-icon-tooltip ${item.grade || 'normal'}"
                                 onerror="this.src='/assets/images/ui/placeholder-item.svg';">
                        </div>
                        <div class="item-name-section">
                            <h4 class="item-name ${item.grade || 'normal'}">
                                ${item.name}
                                ${item.opt_level > 0 ? `<span class="opt-level">+${item.opt_level}</span>` : ''}
                            </h4>
                            <div class="item-type">
                                <span class="rarity-badge ${item.grade || 'normal'}">
                                    ${item.grade || 'Normal'}
                                </span>
                            </div>
                        </div>
                    </div>
                    <hr class="tooltip-separator">
                </div>
                <div class="tooltip-body legacy-content">
                    ${htmlContent}
                </div>
            </div>
        `;
    },

    getLoadingHTML(slotName) {
        const item = this.processedItems[this.currentSlot];
        return `
            <div class="item-tooltip inventory-tooltip tooltip-loading">
                <div class="tooltip-header-new">
                    <div class="icon-and-name">
                        <div class="tooltip-icon">
                            <img src="${item.image}"
                                 alt="${item.name}"
                                 class="item-icon-tooltip ${item.grade || 'normal'}"
                                 onerror="this.src='/assets/images/ui/placeholder-item.svg';">
                        </div>
                        <div class="item-name-section">
                            <h4 class="item-name ${item.grade || 'normal'}">
                                ${item.name}
                                ${item.opt_level > 0 ? `<span class="opt-level">+${item.opt_level}</span>` : ''}
                            </h4>
                            <div class="loading-indicator">
                                <i class="fas fa-spinner fa-spin"></i>
                                <span>Loading details...</span>
                            </div>
                        </div>
                    </div>
                    <hr class="tooltip-separator">
                </div>
            </div>
        `;
    },

    getErrorHTML(slotName, errorMessage) {
        const item = this.processedItems[this.currentSlot];
        return `
            <div class="item-tooltip inventory-tooltip tooltip-error">
                <div class="tooltip-header-new">
                    <div class="icon-and-name">
                        <div class="tooltip-icon">
                            <img src="${item.image}"
                                 alt="${item.name}"
                                 class="item-icon-tooltip ${item.grade || 'normal'}"
                                 onerror="this.src='/assets/images/ui/placeholder-item.svg';">
                        </div>
                        <div class="item-name-section">
                            <h4 class="item-name ${item.grade || 'normal'}">
                                ${item.name}
                                ${item.opt_level > 0 ? `<span class="opt-level">+${item.opt_level}</span>` : ''}
                            </h4>
                            <div class="error-message">
                                <i class="fas fa-exclamation-triangle text-red-400"></i>
                                <span class="text-red-300">${errorMessage}</span>
                            </div>
                        </div>
                    </div>
                    <hr class="tooltip-separator">
                </div>
            </div>
        `;
    },




    /**
     * Get CSS class for magic option type
     */
    getMagOptClass(type) {
        const classMap = {
            'damage': 'mag-opt-damage',
            'defense': 'mag-opt-defense',
            'hit_rate': 'mag-opt-accuracy',
            'dodge_rate': 'mag-opt-evasion',
            'critical_rate': 'mag-opt-critical',
            'hp': 'mag-opt-hp',
            'mp': 'mag-opt-mp',
            'str': 'mag-opt-str',
            'int': 'mag-opt-int',
            'resistance': 'mag-opt-resistance',
            'absorb_damage': 'mag-opt-absorb',
            'damage_increase': 'mag-opt-damage-boost',
            // Common SRO MagOpt attributes
            'MATTR_STR': 'mag-opt-str',
            'MATTR_INT': 'mag-opt-int',
            'MATTR_HP': 'mag-opt-hp',
            'MATTR_MP': 'mag-opt-mp',
            'MATTR_DUR': 'mag-opt-durability',
            'MATTR_REPAIR': 'mag-opt-repair',
            'MATTR_DEC_MAXDUR': 'mag-opt-durability-decrease',
            'MATTR_PHYATT': 'mag-opt-physical-attack',
            'MATTR_MAGATT': 'mag-opt-magical-attack',
            'MATTR_PHYDEF': 'mag-opt-physical-defense',
            'MATTR_MAGDEF': 'mag-opt-magical-defense',
            'MATTR_HR': 'mag-opt-hit-rate',
            'MATTR_ER': 'mag-opt-evasion',
            'MATTR_ATKSPD': 'mag-opt-attack-speed',
            'MATTR_MOVSPD': 'mag-opt-movement-speed'
        };

        return classMap[type] || 'mag-opt-default';
    },

    /**
     * Get icon for magic option type
     */
    getMagOptIcon(type) {
        const iconMap = {
            'MATTR_STR': '<i class="fas fa-fist-raised"></i>',
            'MATTR_INT': '<i class="fas fa-brain"></i>',
            'MATTR_HP': '<i class="fas fa-heart"></i>',
            'MATTR_MP': '<i class="fas fa-tint"></i>',
            'MATTR_DUR': '<i class="fas fa-shield-alt"></i>',
            'MATTR_REPAIR': '<i class="fas fa-tools"></i>',
            'MATTR_DEC_MAXDUR': '<i class="fas fa-shield-virus"></i>',
            'MATTR_PHYATT': '<i class="fas fa-sword"></i>',
            'MATTR_MAGATT': '<i class="fas fa-magic"></i>',
            'MATTR_PHYDEF': '<i class="fas fa-shield"></i>',
            'MATTR_MAGDEF': '<i class="fas fa-magic"></i>',
            'MATTR_HR': '<i class="fas fa-crosshairs"></i>',
            'MATTR_ER': '<i class="fas fa-wind"></i>',
            'MATTR_ATKSPD': '<i class="fas fa-bolt"></i>',
            'MATTR_MOVSPD': '<i class="fas fa-running"></i>',
            'damage': '<i class="fas fa-sword"></i>',
            'defense': '<i class="fas fa-shield"></i>',
            'hp': '<i class="fas fa-heart"></i>',
            'mp': '<i class="fas fa-tint"></i>'
        };

        return iconMap[type] || '<i class="fas fa-star"></i>';
    },

    /**
     * Get description for magic option type
     */
    getMagOptDescription(type) {
        const descMap = {
            'MATTR_STR': 'Physical Strength',
            'MATTR_INT': 'Intelligence Power',
            'MATTR_HP': 'Health Points',
            'MATTR_MP': 'Mana Points',
            'MATTR_DUR': 'Durability Enhancement',
            'MATTR_REPAIR': 'Self-Repair Ability',
            'MATTR_DEC_MAXDUR': 'Durability Reduction',
            'MATTR_PHYATT': 'Physical Attack',
            'MATTR_MAGATT': 'Magical Attack',
            'MATTR_PHYDEF': 'Physical Defense',
            'MATTR_MAGDEF': 'Magical Defense',
            'MATTR_HR': 'Hit Rate Accuracy',
            'MATTR_ER': 'Evasion Rate',
            'MATTR_ATKSPD': 'Attack Speed',
            'MATTR_MOVSPD': 'Movement Speed',
            'damage': 'Damage Enhancement',
            'defense': 'Defense Boost',
            'hp': 'Health Boost',
            'mp': 'Mana Enhancement'
        };

        return descMap[type] || 'Special Enhancement';
    },

    /**
     * Get test blue stats for demonstration
     */
    getTestBlueStats() {
        return [
            {
                'id': 1,
                'name': 'Strength +25',
                'display_name': 'Strength +25',
                'description': 'Increases physical strength',
                'color': '#50cecd',
                'sort_key': 1,
                'is_negative': false,
                'category': 'attribute',
                'icon': 'fas fa-fist-raised'
            },
            {
                'id': 3,
                'name': 'HP +150',
                'display_name': 'HP +150',
                'description': 'Increases health points',
                'color': '#50cecd',
                'sort_key': 3,
                'is_negative': false,
                'category': 'vitality',
                'icon': 'fas fa-heart'
            },
            {
                'id': 7,
                'name': 'Maximum durability 10% decrease',
                'display_name': 'Maximum durability 10% decrease',
                'description': 'Decreases maximum durability',
                'color': '#ff2f51',
                'sort_key': 7,
                'is_negative': true,
                'category': 'penalty',
                'icon': 'fas fa-exclamation-triangle'
            }
        ];
    },

    /**
     * Get test white stats for demonstration
     */
    getTestWhiteStats() {
        return [
            {
                'type': 'physical_attack',
                'display_name': 'Physical Attack Power',
                'min_value': 120,
                'max_value': 180,
                'bonus_percentage': 25,
                'formatted_value': '120 ~ 180',
                'category': 'combat',
                'icon': 'fas fa-sword',
                'raw_text': 'Phy. atk. pwr. 120 ~ 180 (+25%)'
            },
            {
                'type': 'durability',
                'display_name': 'Durability',
                'current_value': 45,
                'max_value': 50,
                'bonus_percentage': 20,
                'formatted_value': '45/50',
                'category': 'utility',
                'icon': 'fas fa-hammer',
                'raw_text': 'Durability 45/50 (+20%)'
            },
            {
                'type': 'attack_rating',
                'display_name': 'Attack Rating',
                'value': 150,
                'bonus_percentage': 30,
                'formatted_value': '150',
                'category': 'combat',
                'icon': 'fas fa-crosshairs',
                'raw_text': 'Attack rating 150 (+30%)'
            }
        ];
    },

    // Helper methods for BlueStat categories and icons
    getBlueStatCategoryName(category) {
        const categoryNames = {
            'attribute': 'Attributes',
            'vitality': 'Vitality & Health',
            'combat': 'Combat Stats',
            'defense': 'Defense',
            'utility': 'Utility',
            'penalty': 'Penalties'
        };
        return categoryNames[category] || 'Other';
    },

    getBlueStatCategoryIcon(category) {
        const categoryIcons = {
            'attribute': 'fas fa-star',
            'vitality': 'fas fa-heart',
            'combat': 'fas fa-sword',
            'defense': 'fas fa-shield-alt',
            'utility': 'fas fa-tools',
            'penalty': 'fas fa-exclamation-triangle'
        };
        return categoryIcons[category] || 'fas fa-question';
    },

    // Helper methods for WhiteStat categories and icons
    getWhiteStatCategoryName(category) {
        const categoryNames = {
            'attribute': 'Base Attributes',
            'vitality': 'Health & Mana',
            'combat': 'Combat Enhancement',
            'defense': 'Protection',
            'utility': 'Special Effects',
            'penalty': 'Drawbacks'
        };
        return categoryNames[category] || 'Other';
    },

    getWhiteStatCategoryIcon(category) {
        const categoryIcons = {
            'attribute': 'fas fa-star-of-life',
            'vitality': 'fas fa-heart-pulse',
            'combat': 'fas fa-crosshairs',
            'defense': 'fas fa-shield',
            'utility': 'fas fa-magic',
            'penalty': 'fas fa-minus-circle'
        };
        return categoryIcons[category] || 'fas fa-question-circle';
    },

    // Helper method for individual WhiteStat icons
    getWhiteStatIcon(type) {
        const statIcons = {
            // Attack Stats
            'phy_attack': 'fas fa-sword',
            'mag_attack': 'fas fa-magic',

            // Defense Stats
            'phy_defense': 'fas fa-shield-alt',
            'mag_defense': 'fas fa-shield',

            // Utility Stats
            'durability': 'fas fa-hammer',
            'attack_distance': 'fas fa-arrows-alt-h',
            'attack_rate': 'fas fa-crosshairs',
            'critical': 'fas fa-bullseye',
            'blocking': 'fas fa-stop',
            'parry_rate': 'fas fa-hand-paper',

            // Absorption Stats
            'phy_absorption': 'fas fa-compress-arrows-alt',
            'mag_absorption': 'fas fa-compress',

            // Reinforcement Stats
            'phy_reinforce': 'fas fa-plus-square',
            'mag_reinforce': 'fas fa-plus-circle',

            // Legacy support
            'strength': 'fas fa-fist-raised',
            'intelligence': 'fas fa-brain',
            'health': 'fas fa-heart',
            'mana': 'fas fa-tint',
            'defense': 'fas fa-shield-alt',
            'attack_rating': 'fas fa-crosshairs',
            'critical_chance': 'fas fa-bullseye',
            'damage': 'fas fa-sword',
            'speed': 'fas fa-tachometer-alt'
        };
        return statIcons[type] || 'fas fa-chart-bar';
    },

    /**
     * Render binding options section
     *
     * @param {Object} bindingOptions - The binding options data
     * @return {string} HTML for binding options display
     */
    renderBindingOptions(bindingOptions) {
        if (!bindingOptions || !bindingOptions.has_binding) {
            return '';
        }

        const glowLevelName = this.getGlowLevelName(bindingOptions.glow_level);
        const glowClass = this.getGlowClass(bindingOptions.glow_level);

        let optionsHtml = '';
        if (bindingOptions.options && bindingOptions.options.length > 0) {
            optionsHtml = bindingOptions.options.map(option => `
                <div class="binding-option-item">
                    <div class="option-badge ${glowClass}">
                        <i class="fas fa-seal"></i>
                        <span class="option-type">${option.type_name}</span>
                    </div>
                    <div class="option-details">
                        <div class="option-slot">Slot ${option.slot}</div>
                        <div class="option-level">Level ${option.opt_level}</div>
                        <div class="option-value ${glowClass}">Value: ${option.opt_value}</div>
                    </div>
                </div>
            `).join('');
        }

        return `
            <div class="binding-options-section ${glowClass}">
                <h5 class="section-title binding-title">
                    <i class="fas fa-magic"></i>
                    <span class="title-text">Binding Options</span>
                    <div class="binding-summary ${glowClass}">
                        <span class="glow-level-badge level-${bindingOptions.glow_level}">${glowLevelName}</span>
                        <span class="binding-count">${bindingOptions.options ? bindingOptions.options.length : 0} Seals</span>
                    </div>
                </h5>
                <div class="binding-options-container">
                    ${optionsHtml || `
                        <div class="binding-placeholder ${glowClass}">
                            <i class="fas fa-star"></i>
                            <span>Enhanced with ${glowLevelName} power</span>
                        </div>
                    `}
                </div>
            </div>
        `;
    },

    /**
     * Get glow level name based on numeric level
     *
     * @param {number} level - The glow level (1-4)
     * @return {string} Human readable glow level name
     */
    getGlowLevelName(level) {
        const names = {
            1: 'Seal of Star',
            2: 'Seal of Moon',
            3: 'Seal of Sun',
            4: 'Seal of Heavy Storm'
        };
        return names[level] || 'Enhanced';
    },

    /**
     * Get CSS glow class based on nOptValue level (1-4)
     *
     * @param {number} glowLevel - The glow level from binding options (0-4)
     * @return {string} CSS class for glow effect
     */
    getGlowClass(glowLevel) {
        if (!glowLevel || glowLevel <= 0) {
            return '';
        }

        const glowClasses = {
            1: 'binding-glow-1',    // Simple glow - light blue
            2: 'binding-glow-2',    // Moderate glow - blue
            3: 'binding-glow-3',    // Strong glow - purple
            4: 'binding-glow-4'     // Intense glow - gold/orange
        };

        return glowClasses[Math.min(glowLevel, 4)] || '';
    }
}));
