// Enhanced Hero Section Interactions - Mobile Optimized
document.addEventListener('DOMContentLoaded', function() {
    const heroSection = document.getElementById('heroSection');
    const heroLogo = document.querySelector('.hero-logo');
    const heroButtons = document.querySelectorAll('.hero-btn');
    const isMobile = window.innerWidth <= 768;
    const isTouch = 'ontouchstart' in window || navigator.maxTouchPoints > 0;

    // Track cleanup
    let isDestroyed = false;
    let activeTimeouts = [];
    let activeIntervals = [];
    let resizeTimeout = null;
    let particleCreateTimeout = null;

    // Helper function to add tracked timeout
    function addTimeout(callback, delay) {
        const timeoutId = setTimeout(() => {
            if (!isDestroyed) callback();
        }, delay);
        activeTimeouts.push(timeoutId);
        return timeoutId;
    }

    // Helper function to add tracked interval
    function addInterval(callback, delay) {
        const intervalId = setInterval(() => {
            if (isDestroyed) {
                clearInterval(intervalId);
                return;
            }
            callback();
        }, delay);
        activeIntervals.push(intervalId);
        return intervalId;
    }

    // Initialize hero section with mobile optimization
    addTimeout(() => {
        if (heroSection && !isDestroyed) {
            heroSection.classList.remove('loading');
            heroSection.classList.add('loaded');
        }
    }, 300);

    // Mobile-specific optimizations
    if (isMobile) {
        // Add mobile class for specific styling
        if (heroSection) {
            heroSection.classList.add('mobile-optimized');
        }

        // Reduce animation complexity on mobile
        document.documentElement.style.setProperty('--animation-duration', '8s');

        // Optimize viewport handling
        const viewport = document.querySelector('meta[name=viewport]');
        if (viewport) {
            viewport.setAttribute('content', 'width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no');
        }
    }

    // Enhanced button interactions with mobile optimization
    heroButtons.forEach(button => {
        // Prevent default behavior and add custom handling
        button.addEventListener('click', function(e) {
            e.preventDefault();

            // Add loading state
            this.classList.add('loading');

            // Create optimized ripple effect
            const ripple = document.createElement('div');
            const rect = this.getBoundingClientRect();
            const size = Math.max(rect.width, rect.height) * (isMobile ? 1.2 : 2);

            // Use touch coordinates if available, otherwise mouse coordinates
            const clientX = e.touches ? e.touches[0].clientX : e.clientX;
            const clientY = e.touches ? e.touches[0].clientY : e.clientY;

            const x = clientX - rect.left - size / 2;
            const y = clientY - rect.top - size / 2;

            ripple.style.cssText = `
                position: absolute;
                width: ${size}px;
                height: ${size}px;
                left: ${x}px;
                top: ${y}px;
                background: radial-gradient(circle,
                    rgba(255, 255, 255, 0.8) 0%,
                    rgba(255, 255, 255, 0.4) 30%,
                    transparent 70%);
                border-radius: 50%;
                transform: scale(0);
                animation: optimizedRipple 0.6s cubic-bezier(0.25, 0.46, 0.45, 0.94);
                pointer-events: none;
                z-index: 10;
            `;

            this.appendChild(ripple);

            // Create mobile-optimized particle effect
            if (!isMobile || window.innerWidth > 480) {
                createParticleBurst.call(this, clientX - rect.left, clientY - rect.top);
            }

            // Add haptic feedback for mobile
            if (isTouch && navigator.vibrate) {
                navigator.vibrate(50);
            }

            // Handle action completion
            setTimeout(() => {
                this.classList.remove('loading');
                this.classList.add('success');

                // Execute action
                if (this.getAttribute('onclick')) {
                    try {
                        // Safer alternative to eval
                        const onclickCode = this.getAttribute('onclick');
                        if (onclickCode.includes('window.open')) {
                            // Handle window.open calls safely
                            const urlMatch = onclickCode.match(/window\.open\(['"`]([^'"`]+)['"`]/);
                            if (urlMatch) {
                                window.open(urlMatch[1], '_blank');
                            }
                        } else if (onclickCode.includes('location.href')) {
                            // Handle location.href safely
                            const urlMatch = onclickCode.match(/location\.href\s*=\s*['"`]([^'"`]+)['"`]/);
                            if (urlMatch) {
                                window.location.href = urlMatch[1];
                            }
                        }
                    } catch (e) {
                        // Error executing onclick
                    }
                } else if (this.href && this.href !== '#') {
                    window.location.href = this.href;
                }

                setTimeout(() => {
                    this.classList.remove('success');
                }, 600);
            }, 600);

            setTimeout(() => {
                ripple.remove();
            }, 600);
        });

        // Enhanced mobile touch interactions
        if (isTouch) {
            let touchStartTime;

            button.addEventListener('touchstart', function(e) {
                e.preventDefault();
                touchStartTime = Date.now();
                this.classList.add('touch-active');
                this.style.transform = 'scale(0.95)';
                this.style.filter = 'brightness(1.1)';
            }, { passive: false });

            button.addEventListener('touchend', function(e) {
                e.preventDefault();
                const touchDuration = Date.now() - touchStartTime;

                this.classList.remove('touch-active');
                this.style.transform = '';
                this.style.filter = '';

                // Only trigger click if it was a quick tap
                if (touchDuration < 500) {
                    this.click();
                }
            }, { passive: false });

            button.addEventListener('touchmove', function(e) {
                // Cancel touch if user moves finger too much
                const touch = e.touches[0];
                const rect = this.getBoundingClientRect();

                if (touch.clientX < rect.left || touch.clientX > rect.right ||
                    touch.clientY < rect.top || touch.clientY > rect.bottom) {
                    this.classList.remove('touch-active');
                    this.style.transform = '';
                    this.style.filter = '';
                }
            });
        } else {
            // Desktop hover effects
            button.addEventListener('mouseenter', function() {
                this.style.transform = 'translateY(-5px) scale(1.05)';
                this.style.filter = 'brightness(1.1)';
            });

            button.addEventListener('mouseleave', function() {
                this.style.transform = '';
                this.style.filter = '';
            });
        }
    });

    // Particle burst creation function
    function createParticleBurst(x, y) {
        const colors = this.classList.contains('hero-btn-register')
            ? ['#667eea', '#764ba2', '#f093fb', '#f5576c']
            : ['#11998e', '#38ef7d', '#2196f3', '#21cbf3'];

        const particleCount = isMobile ? 4 : 8;

        for (let i = 0; i < particleCount; i++) {
            const particle = document.createElement('div');
            const angle = (i / particleCount) * Math.PI * 2;
            const velocity = isMobile ? 40 + Math.random() * 20 : 60 + Math.random() * 30;
            const size = isMobile ? 2 + Math.random() * 2 : 3 + Math.random() * 3;

            particle.style.cssText = `
                position: absolute;
                width: ${size}px;
                height: ${size}px;
                background: ${colors[Math.floor(Math.random() * colors.length)]};
                border-radius: 50%;
                left: ${x}px;
                top: ${y}px;
                pointer-events: none;
                z-index: 15;
                box-shadow: 0 0 ${size * 2}px currentColor;
            `;

            this.appendChild(particle);

            const moveX = Math.cos(angle) * velocity;
            const moveY = Math.sin(angle) * velocity;

            particle.animate([
                { transform: 'translate(0, 0) scale(1)', opacity: 1 },
                { transform: `translate(${moveX}px, ${moveY}px) scale(0)`, opacity: 0 }
            ], {
                duration: isMobile ? 600 : 800,
                easing: 'cubic-bezier(0.25, 0.46, 0.45, 0.94)'
            }).onfinish = () => particle.remove();
        }
    }

    // Enhanced logo interactions with mobile optimization
    if (heroLogo) {
        if (isTouch) {
            heroLogo.addEventListener('touchstart', function() {
                this.style.transform = 'scale(1.05)';
                this.style.filter = 'brightness(1.2) saturate(1.2)';
            });

            heroLogo.addEventListener('touchend', function() {
                this.style.transform = '';
                this.style.filter = '';

                // Add spin animation on tap
                this.style.animation = 'logoSpin 1s ease-in-out';
                setTimeout(() => {
                    this.style.animation = '';
                }, 1000);
            });
        } else {
            heroLogo.addEventListener('mouseenter', function() {
                this.style.transform = 'scale(1.1) rotate(5deg)';
                this.style.filter = 'brightness(1.2) saturate(1.2)';
            });

            heroLogo.addEventListener('mouseleave', function() {
                this.style.transform = '';
                this.style.filter = '';
            });

            heroLogo.addEventListener('click', function() {
                this.style.animation = 'logoSpin 1s ease-in-out';
                setTimeout(() => {
                    this.style.animation = '';
                }, 1000);
            });
        }
    }

    // Mobile-optimized parallax effect
    if (!isMobile) {
        window.addEventListener('scroll', function() {
            const scrolled = window.pageYOffset;
            const background = document.querySelector('.hero-galaxy-bg');
            if (background) {
                background.style.transform = `translateY(${scrolled * 0.3}px)`;
            }
        });
    }

    // Dynamic particles creation with mobile optimization and visibility check
    function createHeroParticles() {
        // Stop creating particles if page is hidden or destroyed
        if (document.hidden || isDestroyed) {
            return;
        }

        const particlesContainer = document.querySelector('.hero-particles');
        if (!particlesContainer) return;

        particlesContainer.innerHTML = '';

        let particleCount;
        if (window.innerWidth <= 360) {
            particleCount = 3;
        } else if (window.innerWidth <= 480) {
            particleCount = 4;
        } else if (window.innerWidth <= 768) {
            particleCount = 6;
        } else {
            particleCount = 9;
        }

        for (let i = 0; i < particleCount; i++) {
            if (isDestroyed) break; // Check during loop

            const particle = document.createElement('div');
            particle.className = 'hero-particle';
            particle.style.left = Math.random() * 100 + '%';
            particle.style.animationDelay = Math.random() * 12 + 's';

            const duration = isMobile ? (6 + Math.random() * 4) : (10 + Math.random() * 8);
            particle.style.animationDuration = duration + 's';

            // Add different sizes for visual variety
            const size = isMobile ?
                (Math.random() > 0.5 ? '2px' : '1px') :
                (Math.random() > 0.6 ? '3px' : '2px');
            particle.style.width = size;
            particle.style.height = size;

            particlesContainer.appendChild(particle);
        }
    }

    // Initialize particles
    createHeroParticles();

    // Handle window resize and orientation change
    function handleResize() {
        clearTimeout(resizeTimeout);
        resizeTimeout = addTimeout(() => {
            createHeroParticles();
        }, 250);
    }

    window.addEventListener('resize', handleResize);
    window.addEventListener('orientationchange', function() {
        setTimeout(handleResize, 300);
    });

    // RTL Support with mobile optimization
    function updateHeroRTL() {
        const isRTL = document.documentElement.dir === 'rtl';
        const heroContent = document.querySelector('.hero-content');

        if (isRTL && heroContent) {
            heroContent.style.direction = 'rtl';

            if (isMobile) {
                heroContent.style.textAlign = 'center';
            }
        }
    }

    // Listen for language changes
    const observer = new MutationObserver(function(mutations) {
        mutations.forEach(function(mutation) {
            if (mutation.type === 'attributes' && mutation.attributeName === 'dir') {
                updateHeroRTL();
            }
        });
    });

    observer.observe(document.documentElement, {
        attributes: true,
        attributeFilter: ['dir']
    });

    updateHeroRTL();

    // Enhanced entrance animation with mobile optimization
    const heroElements = document.querySelectorAll('.hero-logo, .hero-btn, .gaming-player-counter');

    heroElements.forEach((element, index) => {
        element.style.opacity = '0';
        element.style.transform = isMobile ? 'translateY(20px)' : 'translateY(30px)';
        element.style.transition = 'all 0.8s cubic-bezier(0.68, -0.55, 0.265, 1.55)';

        const delay = isMobile ? 200 + (index * 100) : 400 + (index * 150);
        addTimeout(() => {
            element.style.opacity = '1';
            element.style.transform = 'translateY(0)';
        }, delay);
    });

    // Handle visibility change to pause animations when tab is not active
    document.addEventListener('visibilitychange', function() {
        const heroSection = document.querySelector('.hero-section');

        if (document.hidden) {
            if (heroSection) {
                heroSection.style.animationPlayState = 'paused';
            }
            // Clear particle creation timeout if running
            if (particleCreateTimeout) {
                clearTimeout(particleCreateTimeout);
                particleCreateTimeout = null;
            }
        } else if (!isDestroyed) {
            if (heroSection) {
                heroSection.style.animationPlayState = 'running';
            }
            // Recreate particles when page becomes visible again
            particleCreateTimeout = addTimeout(() => {
                createHeroParticles();
            }, 100);
        }
    });

    // Cleanup function
    function cleanup() {
        isDestroyed = true;

        // Clear all timeouts
        activeTimeouts.forEach(timeoutId => {
            clearTimeout(timeoutId);
        });
        activeTimeouts = [];

        // Clear all intervals
        activeIntervals.forEach(intervalId => {
            clearInterval(intervalId);
        });
        activeIntervals = [];

        // Clear specific timeouts
        if (resizeTimeout) {
            clearTimeout(resizeTimeout);
            resizeTimeout = null;
        }

        if (particleCreateTimeout) {
            clearTimeout(particleCreateTimeout);
            particleCreateTimeout = null;
        }

        // Hero section cleanup completed
    }

    // Add cleanup listener
    window.addEventListener('beforeunload', cleanup);

    // Mobile performance optimizations
    if (isMobile) {
        // Reduce animation complexity on mobile
        const style = document.createElement('style');
        style.textContent = `
            @media (max-width: 768px) {
                .hero-btn {
                    will-change: transform;
                    backface-visibility: hidden;
                    -webkit-transform-style: preserve-3d;
                }

                .btn-bg-effect {
                    animation-duration: 4s;
                }

                .btn-particles .particle {
                    animation-duration: 4s !important;
                }

                /* Reduce gradient animation frequency on mobile */
                .hero-btn-register,
                .hero-btn-download {
                    animation-duration: 8s;
                }
            }
        `;
        document.head.appendChild(style);
    }

    // Device orientation change handler
    window.addEventListener('orientationchange', function() {
        addTimeout(() => {
            if (!isDestroyed) {
                createHeroParticles();
                // Recalculate layout if needed
                const heroContent = document.querySelector('.hero-content');
                if (heroContent) {
                    heroContent.style.minHeight = window.innerHeight * 0.6 + 'px';
                }
            }
        }, 300);
    });
});

// Add advanced animation CSS
const heroStyle = document.createElement('style');
heroStyle.textContent = `
    @keyframes optimizedRipple {
        0% {
            transform: scale(0);
            opacity: 0.8;
        }
        50% {
            transform: scale(0.5);
            opacity: 0.6;
        }
        100% {
            transform: scale(1);
            opacity: 0;
        }
    }

    @keyframes logoSpin {
        0% { transform: scale(1) rotate(0deg); }
        50% { transform: scale(1.2) rotate(180deg); }
        100% { transform: scale(1) rotate(360deg); }
    }

    @keyframes pulseGlow {
        0%, 100% {
            box-shadow: 0 0 20px rgba(255, 255, 255, 0.3);
        }
        50% {
            box-shadow: 0 0 40px rgba(255, 255, 255, 0.6);
        }
    }

    .hero-btn {
        position: relative;
        overflow: hidden;
    }

    .hero-btn:hover {
        animation: pulseGlow 2s ease-in-out infinite;
    }

    /* Advanced floating animation for better visual appeal */
    @keyframes floatUpDown {
        0%, 100% {
            transform: translateY(0px);
        }
        50% {
            transform: translateY(-10px);
        }
    }

    .hero-btn:not(:hover) {
        animation: floatUpDown 4s ease-in-out infinite;
    }

    /* Mobile-specific floating animation */
    @media (max-width: 768px) {
        .hero-btn:not(:hover) {
            animation: mobileFloat 3s ease-in-out infinite;
        }

        @keyframes mobileFloat {
            0%, 100% {
                transform: translateY(0px);
            }
            50% {
                transform: translateY(-5px);
            }
        }
    }

    /* Mobile performance optimizations */
    @media (max-width: 768px) {
        .hero-btn {
            -webkit-tap-highlight-color: transparent;
            touch-action: manipulation;
        }

        .hero-btn.touch-active {
            transition: transform 0.1s ease;
        }

        /* Simplified animations for mobile */
        .hero-galaxy-bg,
        .hero-logo::before {
            animation-duration: 15s;
        }

        .hero-particle {
            animation-duration: 8s;
        }
    }

    /* Enhanced mobile accessibility */
    @media (max-width: 480px) {
        .hero-btn {
            min-height: 48px;
            touch-action: manipulation;
            -webkit-user-select: none;
            user-select: none;
        }
    }
`;
document.head.appendChild(heroStyle);

// Add mobile-specific CSS for better performance
if (window.innerWidth <= 768) {
    const mobileCSS = document.createElement('style');
    mobileCSS.textContent = `
        .hero-section.mobile-optimized {
            background-attachment: scroll;
            transform: translate3d(0,0,0);
            -webkit-transform: translate3d(0,0,0);
        }

        .hero-logo,
        .hero-btn,
        .gaming-player-counter {
            transform: translate3d(0,0,0);
            -webkit-transform: translate3d(0,0,0);
            backface-visibility: hidden;
            -webkit-backface-visibility: hidden;
        }

        /* Reduced motion for low-end devices */
        @media (max-width: 480px) and (max-height: 800px) {
            .hero-galaxy-bg::before {
                animation: none;
                opacity: 0.3;
            }

            .hero-logo::before {
                animation-duration: 30s;
            }
        }
    `;
    document.head.appendChild(mobileCSS);
}


