// Header Tablet JavaScript Functionality
document.addEventListener('DOMContentLoaded', function() {
    // Initialize all tablet header functionality
    initTabletHeader();
});

function initTabletHeader() {
    // Initialize dropdowns
    initTabletDropdowns();

    // Initialize notifications
    initTabletNotifications();

    // Initialize user menu
    initTabletUserMenu();

    // Initialize menu interactions (active states)
    initTabletMenuInteractions();

    // Add keyboard navigation
    initTabletKeyboardNavigation();

    // Add scroll effects
    initTabletScrollEffects();
}

function initTabletDropdowns() {
    // Handle main navigation dropdown
    const dropdownTriggers = document.querySelectorAll('.dropdown-trigger-tablet');

    dropdownTriggers.forEach(trigger => {
        trigger.addEventListener('click', function(e) {
            e.preventDefault();
            const dropdown = this.closest('.dropdown-tablet');
            const content = dropdown.querySelector('.dropdown-content-tablet');

            // Close other dropdowns
            document.querySelectorAll('.dropdown-content-tablet').forEach(other => {
                if (other !== content) {
                    other.classList.remove('show');
                }
            });

            // Toggle current dropdown
            content.classList.toggle('show');
        });
    });

    // Close dropdowns when clicking outside
    document.addEventListener('click', function(e) {
        if (!e.target.closest('.dropdown-tablet')) {
            document.querySelectorAll('.dropdown-content-tablet').forEach(dropdown => {
                dropdown.classList.remove('show');
            });
        }
    });
}

function initTabletNotifications() {
    // Notification buttons
    const notificationButtons = {
        notifications: document.getElementById('tabletNotificationsBtn'),
        adminMessages: document.getElementById('tabletAdminMessagesBtn'),
        charges: document.getElementById('tabletChargesBtn')
    };

    // Notification dropdowns
    const notificationDropdowns = {
        notifications: document.getElementById('tabletNotificationsDropdown'),
        adminMessages: document.getElementById('tabletAdminMessagesDropdown'),
        charges: document.getElementById('tabletChargesDropdown')
    };

    // Add click handlers for notification buttons
    Object.keys(notificationButtons).forEach(key => {
        const button = notificationButtons[key];
        const dropdown = notificationDropdowns[key];

        if (button && dropdown) {
            button.addEventListener('click', function(e) {
                e.stopPropagation();

                // Close other notification dropdowns
                Object.values(notificationDropdowns).forEach(other => {
                    if (other && other !== dropdown && other.classList) {
                        other.classList.remove('show');
                    }
                });

                // Close user menu
                const userMenu = document.querySelector('.user-dropdown-menu-tablet');
                if (userMenu && userMenu.classList) {
                    userMenu.classList.remove('show');
                }

                // Toggle current dropdown
                if (dropdown.classList) {
                    dropdown.classList.toggle('show');
                }
            });
        }
    });

    // Close notifications when clicking outside
    document.addEventListener('click', function(e) {
        if (!e.target.closest('.notification-icon-container-tablet')) {
            Object.values(notificationDropdowns).forEach(dropdown => {
                if (dropdown && dropdown.classList) {
                    dropdown.classList.remove('show');
                }
            });
        }
    });

    // Listen for Livewire notification updates
    if (window.Livewire) {
        Livewire.on('notification-updated', (data) => {
            if (data.type === 'tablet') {
                updateTabletNotificationBadge('tabletNotificationsBadge', data.unreadCount);
            }
        });
    }

    // Load notification data (simulated)
    loadTabletNotificationData();
}

function initTabletUserMenu() {
    const userTrigger = document.querySelector('.user-dropdown-trigger-tablet');
    const userMenu = document.querySelector('.user-dropdown-menu-tablet');

    if (userTrigger && userMenu) {
        // Add click handler
        userTrigger.addEventListener('click', function(e) {
            e.stopPropagation();

            // Close notification dropdowns
            document.querySelectorAll('.notification-dropdown-tablet').forEach(dropdown => {
                if (dropdown && dropdown.classList) {
                    dropdown.classList.remove('show');
                }
            });

            // Toggle user menu with animation
            const isVisible = userMenu.classList.contains('show');
            if (isVisible) {
                // Hide menu
                userMenu.classList.remove('show');
                setTimeout(() => {
                    if (userMenu.parentNode) {
                        userMenu.style.display = 'none';
                    }
                }, 300); // Match transition duration
            } else {
                // Show menu
                userMenu.style.display = 'block';
                setTimeout(() => {
                    if (userMenu.classList) {
                        userMenu.classList.add('show');
                    }
                }, 10); // Small delay to allow display change

                // Initialize tabs on first open
                if (!userMenu.hasAttribute('data-initialized')) {
                    initTabletMenuTabs();
                    userMenu.setAttribute('data-initialized', 'true');
                }
            }
        });

        // Add hover effects for better UX
        userTrigger.addEventListener('mouseenter', function() {
            this.style.transform = 'translateY(-2px)';
        });

        userTrigger.addEventListener('mouseleave', function() {
            this.style.transform = 'translateY(0)';
        });
    }

    // Close user menu when clicking outside
    document.addEventListener('click', function(e) {
        if (!e.target.closest('.user-dropdown-container-tablet')) {
            const userMenu = document.querySelector('.user-dropdown-menu-tablet');
            if (userMenu && userMenu.classList && userMenu.classList.contains('show')) {
                userMenu.classList.remove('show');
                setTimeout(() => {
                    if (userMenu.parentNode) {
                        userMenu.style.display = 'none';
                    }
                }, 300);
            }
        }
    });

    // Add keyboard navigation for user menu
    document.addEventListener('keydown', function(e) {
        const userMenu = document.querySelector('.user-dropdown-menu-tablet');
        if (userMenu && userMenu.classList && userMenu.classList.contains('show')) {
            if (e.key === 'Escape') {
                userMenu.classList.remove('show');
                setTimeout(() => {
                    if (userMenu.parentNode) {
                        userMenu.style.display = 'none';
                    }
                }, 300);
                const trigger = document.querySelector('.user-dropdown-trigger-tablet');
                if (trigger && trigger.focus) {
                    trigger.focus();
                }
            }
        }
    });

    // Initialize balance animations
    initTabletBalanceAnimations();
}

function initTabletBalanceAnimations() {
    // Add hover animations to balance items
    const balanceItems = document.querySelectorAll('.balance-item-tablet');
    balanceItems.forEach(item => {
        item.addEventListener('mouseenter', function() {
            const icon = this.querySelector('.balance-icon-tablet');
            if (icon) {
                icon.style.transform = 'scale(1.1) rotate(5deg)';
                icon.style.transition = 'transform 0.3s ease';
            }
        });

        item.addEventListener('mouseleave', function() {
            const icon = this.querySelector('.balance-icon-tablet');
            if (icon) {
                icon.style.transform = 'scale(1) rotate(0deg)';
            }
        });
    });
}

function initTabletKeyboardNavigation() {
    // Add keyboard navigation for accessibility
    document.addEventListener('keydown', function(e) {
        // ESC key to close all dropdowns
        if (e.key === 'Escape') {
            closeAllTabletDropdowns();
        }

        // Tab navigation
        if (e.key === 'Tab') {
            // Handle focus management
            manageTabletFocus(e);
        }
    });
}

function initTabletScrollEffects() {
    let lastScrollTop = 0;
    let isScrollingDown = false;
    const header = document.querySelector('.header-tablet');
    const scrollThreshold = 50; // Minimum scroll distance to trigger hide/show

    if (header) {
        window.addEventListener('scroll', function() {
            const scrollTop = window.pageYOffset || document.documentElement.scrollTop;

            // Add scrolled class for styling
            if (scrollTop > 50) {
                header.classList.add('scrolled');
            } else {
                header.classList.remove('scrolled');
            }

            // Determine scroll direction
            if (scrollTop > lastScrollTop && scrollTop > scrollThreshold) {
                // Scrolling down
                if (!isScrollingDown) {
                    header.classList.add('hide');
                    header.classList.remove('show');
                    isScrollingDown = true;
                }
            } else if (scrollTop < lastScrollTop) {
                // Scrolling up
                if (isScrollingDown) {
                    header.classList.add('show');
                    header.classList.remove('hide');
                    isScrollingDown = false;
                }
            }

            lastScrollTop = scrollTop;
        });
    }
}

function closeAllTabletDropdowns() {
    // Close all dropdowns
    document.querySelectorAll('.dropdown-content-tablet, .notification-dropdown-tablet, .user-dropdown-menu-tablet').forEach(dropdown => {
        if (dropdown && dropdown.classList) {
            dropdown.classList.remove('show');
        }
    });
}

function manageTabletFocus(e) {
    // Focus management for accessibility
    const focusableElements = document.querySelectorAll(
        '.nav-item-tablet, .notification-btn-tablet, .user-dropdown-trigger-tablet, .dropdown-item-tablet, .notification-item-tablet, .dropdown-menu-item-tablet'
    );

    if (focusableElements.length > 0) {
        const firstElement = focusableElements[0];
        const lastElement = focusableElements[focusableElements.length - 1];

        if (e.shiftKey) {
            // Shift + Tab
            if (document.activeElement === firstElement) {
                if (lastElement && lastElement.focus) {
                    lastElement.focus();
                    e.preventDefault();
                }
            }
        } else {
            // Tab
            if (document.activeElement === lastElement) {
                if (firstElement && firstElement.focus) {
                    firstElement.focus();
                    e.preventDefault();
                }
            }
        }
    }
}

function loadTabletNotificationData() {
    // Removed simulated data - notifications are now handled by Livewire
    // Real notification data is loaded via Livewire component
}

function updateTabletNotificationBadge(badgeId, count) {
    const badge = document.getElementById(badgeId);
    if (badge) {
        if (count > 0) {
            badge.textContent = count > 99 ? '99+' : count;
            badge.style.display = 'flex';
        } else {
            badge.style.display = 'none';
        }
    }
}

function loadTabletNotificationItems() {
    // Removed - notification items are now handled by Livewire component
}

function createTabletNotificationItem(notification) {
    // Removed - notification items are now handled by Livewire component
}

// Utility functions
function debounce(func, wait) {
    let timeout;
    return function executedFunction(...args) {
        const later = () => {
            clearTimeout(timeout);
            func(...args);
        };
        clearTimeout(timeout);
        timeout = setTimeout(later, wait);
    };
}

// Export functions for potential use in other scripts
window.HeaderTablet = {
    closeAllTabletDropdowns,
    updateTabletNotificationBadge,
    loadTabletNotificationData
};

function initTabletMenuInteractions() {
    // Add click handlers for menu items
    const menuItems = document.querySelectorAll('.dropdown-menu-item-tablet');
    menuItems.forEach(item => {
        if (item) {
            item.addEventListener('click', function(e) {
                // Remove active class from all items
                menuItems.forEach(other => {
                    if (other && other.classList) {
                        other.classList.remove('active');
                    }
                });
                // Add active class to clicked item
                if (this.classList) {
                    this.classList.add('active');

                    // Add click effect
                    this.style.transform = 'scale(0.95)';
                    setTimeout(() => {
                        this.style.transform = 'scale(1)';
                    }, 150);
                }
            });
        }
    });

    // Add hover effects for notification items
    const notificationItems = document.querySelectorAll('.notification-item-tablet');
    notificationItems.forEach(item => {
        if (item) {
            item.addEventListener('click', function() {
                // Mark as read (visual feedback)
                this.style.opacity = '0.7';
                setTimeout(() => {
                    this.style.opacity = '1';
                }, 200);
            });
        }
    });

    // Handle nav item clicks for active state
    const navItems = document.querySelectorAll('.nav-item-tablet');
    navItems.forEach(item => {
        if (item) {
            item.addEventListener('click', function(e) {
                // Remove active from all nav items
                navItems.forEach(other => {
                    if (other && other.classList) {
                        other.classList.remove('active');
                    }
                });
                // Add active to clicked nav item
                if (this.classList) {
                    this.classList.add('active');
                }
            });
        }
    });

    // Handle dropdown item clicks
    const dropdownItems = document.querySelectorAll('.dropdown-item-tablet');
    dropdownItems.forEach(item => {
        if (item) {
            item.addEventListener('click', function(e) {
                // Remove active from all dropdown items
                dropdownItems.forEach(other => {
                    if (other && other.classList) {
                        other.classList.remove('active');
                    }
                });
                // Add active to clicked dropdown item
                if (this.classList) {
                    this.classList.add('active');
                }
            });
        }
    });

    // Handle quick action buttons
    const quickActionBtns = document.querySelectorAll('.quick-action-btn-tablet');
    quickActionBtns.forEach(btn => {
        if (btn) {
            btn.addEventListener('click', function(e) {
                // Add click animation
                this.style.transform = 'scale(0.95)';
                setTimeout(() => {
                    this.style.transform = 'scale(1)';
                }, 150);
            });
        }
    });

    // Handle support buttons
    const supportBtns = document.querySelectorAll('.support-btn-tablet');
    supportBtns.forEach(btn => {
        if (btn) {
            btn.addEventListener('click', function(e) {
                // Add click animation
                this.style.transform = 'scale(0.95)';
                setTimeout(() => {
                    this.style.transform = 'scale(1)';
                }, 150);
            });
        }
    });

    // Handle setting toggles
    const settingToggles = document.querySelectorAll('.setting-toggle-tablet input');
    settingToggles.forEach(toggle => {
        if (toggle) {
            toggle.addEventListener('change', function() {
                // Add visual feedback
                const slider = this.nextElementSibling;
                if (slider && slider.classList) {
                    if (this.checked) {
                        slider.classList.add('active');
                    } else {
                        slider.classList.remove('active');
                    }
                }

                // Save setting (simulated)
                saveTabletSetting(this.name, this.checked);
            });
        }
    });

    // Handle setting selects
    const settingSelects = document.querySelectorAll('.setting-select-tablet');
    settingSelects.forEach(select => {
        if (select) {
            select.addEventListener('change', function() {
                // Save setting (simulated)
                saveTabletSetting(this.name, this.value);
            });
        }
    });
}

function initTabletMenuTabs() {
    const tabButtons = document.querySelectorAll('.tab-button-tablet');
    const tabPanes = document.querySelectorAll('.tab-pane-tablet');

    tabButtons.forEach(button => {
        if (button) {
            button.addEventListener('click', function() {
                const targetTab = this.getAttribute('data-tab');

                // Remove active class from all buttons and panes
                tabButtons.forEach(btn => {
                    if (btn && btn.classList) {
                        btn.classList.remove('active');
                    }
                });
                tabPanes.forEach(pane => {
                    if (pane && pane.classList) {
                        pane.classList.remove('active');
                    }
                });

                // Add active class to clicked button and corresponding pane
                if (this.classList) {
                    this.classList.add('active');
                }
                const targetPane = document.getElementById(targetTab + '-tab');
                if (targetPane && targetPane.classList) {
                    targetPane.classList.add('active');
                }

                // Add smooth transition effect
                if (targetPane) {
                    targetPane.style.opacity = '0';
                    setTimeout(() => {
                        targetPane.style.opacity = '1';
                    }, 50);
                }
            });
        }
    });
}

function saveTabletSetting(settingName, value) {
    // Simulate saving setting to localStorage or sending to server
    const settings = JSON.parse(localStorage.getItem('tabletUserSettings') || '{}');
    settings[settingName] = value;
    localStorage.setItem('tabletUserSettings', JSON.stringify(settings));

    // Show feedback
    showTabletSettingFeedback('Setting saved successfully!');
}

function showTabletSettingFeedback(message) {
    // Create feedback element
    const feedback = document.createElement('div');
    feedback.className = 'setting-feedback-tablet';
    feedback.textContent = message;

    // Add to menu
    const menu = document.querySelector('.user-dropdown-menu-tablet');
    if (menu) {
        menu.appendChild(feedback);

        // Remove after 3 seconds
        setTimeout(() => {
            if (feedback.parentNode) {
                feedback.remove();
            }
        }, 3000);
    }
}
