// Header Desktop JavaScript Functionality
document.addEventListener('DOMContentLoaded', function() {
    // Initialize all desktop header functionality
    initDesktopHeader();
});

function initDesktopHeader() {
    // Initialize dropdowns
    initDropdowns();

    // Initialize notifications
    initNotifications();

    // Initialize user menu
    initUserMenu();

    // Initialize menu interactions (active states, RTL support)
    initMenuInteractions();

    // Add keyboard navigation
    initKeyboardNavigation();

    // Add scroll effects
    initScrollEffects();
}

function initDropdowns() {
    // Handle main navigation dropdown
    const dropdownTriggers = document.querySelectorAll('.dropdown-trigger-desktop');

    dropdownTriggers.forEach(trigger => {
        if (trigger) {
            trigger.addEventListener('click', function(e) {
                e.preventDefault();
                const dropdown = this.closest('.dropdown-desktop');
                if (dropdown) {
                    const content = dropdown.querySelector('.dropdown-content-desktop');

                    // Close other dropdowns
                    document.querySelectorAll('.dropdown-content-desktop').forEach(other => {
                        if (other && other !== content && other.classList) {
                            other.classList.remove('show');
                        }
                    });

                    // Toggle current dropdown
                    if (content && content.classList) {
                        content.classList.toggle('show');
                    }
                }
            });
        }
    });

    // Close dropdowns when clicking outside
    document.addEventListener('click', function(e) {
        if (!e.target.closest('.dropdown-desktop')) {
            document.querySelectorAll('.dropdown-content-desktop').forEach(dropdown => {
                if (dropdown && dropdown.classList) {
                    dropdown.classList.remove('show');
                }
            });
        }
    });
}

function initNotifications() {
    // Notification buttons
    const notificationButtons = {
        notifications: document.getElementById('notificationsBtn'),
        adminMessages: document.getElementById('adminMessagesBtn'),
        charges: document.getElementById('chargesBtn')
    };

    // Notification dropdowns
    const notificationDropdowns = {
        notifications: document.getElementById('notificationsDropdown'),
        adminMessages: document.getElementById('adminMessagesDropdown'),
        charges: document.getElementById('chargesDropdown')
    };

    // Add click handlers for notification buttons
    Object.keys(notificationButtons).forEach(key => {
        const button = notificationButtons[key];
        const dropdown = notificationDropdowns[key];

        if (button && dropdown) {
            button.addEventListener('click', function(e) {
                e.stopPropagation();

                // Close other notification dropdowns
                Object.values(notificationDropdowns).forEach(other => {
                    if (other && other.classList) {
                        other.classList.remove('show');
                    }
                });

                // Close user menu
                const userMenu = document.querySelector('.user-dropdown-menu-desktop');
                if (userMenu && userMenu.classList) {
                    userMenu.classList.remove('show');
                }

                // Toggle current dropdown
                if (dropdown.classList) {
                    dropdown.classList.toggle('show');
                }
            });
        }
    });

    // Close notifications when clicking outside
    document.addEventListener('click', function(e) {
        if (!e.target.closest('.notification-icon-container-desktop')) {
            Object.values(notificationDropdowns).forEach(dropdown => {
                if (dropdown && dropdown.classList) {
                    dropdown.classList.remove('show');
                }
            });
        }
    });

    // Listen for Livewire notification updates
    if (window.Livewire) {
        Livewire.on('notification-updated', (data) => {
            if (data.type === 'desktop') {
                updateNotificationBadge('notificationsBadge', data.unreadCount);
            }
        });
    }

    // Load notification data (simulated)
    loadNotificationData();
}

function initUserMenu() {
    const userTrigger = document.querySelector('.user-dropdown-trigger-desktop');
    const userMenu = document.querySelector('.user-dropdown-menu-desktop');

    if (userTrigger && userMenu) {
        // Add click handler
        userTrigger.addEventListener('click', function(e) {
            e.stopPropagation();

            // Close notification dropdowns
            document.querySelectorAll('.notification-dropdown-desktop').forEach(dropdown => {
                if (dropdown && dropdown.classList) {
                    dropdown.classList.remove('show');
                }
            });

            // Toggle user menu with animation
            const isVisible = userMenu.classList.contains('show');
            if (isVisible) {
                // Hide menu
                userMenu.classList.remove('show');
                setTimeout(() => {
                    if (userMenu.parentNode) {
                        userMenu.style.display = 'none';
                    }
                }, 300); // Match transition duration
            } else {
                // Show menu
                userMenu.style.display = 'block';
                setTimeout(() => {
                    if (userMenu.classList) {
                        userMenu.classList.add('show');
                    }
                }, 10); // Small delay to allow display change
            }
        });

        // Add hover effects for better UX
        userTrigger.addEventListener('mouseenter', function() {
            this.style.transform = 'translateY(-2px)';
        });

        userTrigger.addEventListener('mouseleave', function() {
            this.style.transform = 'translateY(0)';
        });
    }

    // Close user menu when clicking outside
    document.addEventListener('click', function(e) {
        if (!e.target.closest('.user-dropdown-container-desktop')) {
            const userMenu = document.querySelector('.user-dropdown-menu-desktop');
            if (userMenu && userMenu.classList && userMenu.classList.contains('show')) {
                userMenu.classList.remove('show');
                setTimeout(() => {
                    if (userMenu.parentNode) {
                        userMenu.style.display = 'none';
                    }
                }, 300);
            }
        }
    });

    // Add keyboard navigation for user menu
    document.addEventListener('keydown', function(e) {
        const userMenu = document.querySelector('.user-dropdown-menu-desktop');
        if (userMenu && userMenu.classList && userMenu.classList.contains('show')) {
            if (e.key === 'Escape') {
                userMenu.classList.remove('show');
                setTimeout(() => {
                    if (userMenu.parentNode) {
                        userMenu.style.display = 'none';
                    }
                }, 300);
                const trigger = document.querySelector('.user-dropdown-trigger-desktop');
                if (trigger) {
                    trigger.focus();
                }
            }
        }
    });

    // Initialize balance animations
    initBalanceAnimations();
}

function initBalanceAnimations() {
    // Add hover animations to balance items
    const balanceItems = document.querySelectorAll('.balance-item-desktop');
    balanceItems.forEach(item => {
        if (item) {
            item.addEventListener('mouseenter', function() {
                const icon = this.querySelector('.balance-icon-desktop');
                if (icon) {
                    icon.style.transform = 'scale(1.1) rotate(5deg)';
                    icon.style.transition = 'transform 0.3s ease';
                }
            });

            item.addEventListener('mouseleave', function() {
                const icon = this.querySelector('.balance-icon-desktop');
                if (icon) {
                    icon.style.transform = 'scale(1) rotate(0deg)';
                }
            });
        }
    });
}


function initKeyboardNavigation() {
    // Add keyboard navigation for accessibility
    document.addEventListener('keydown', function(e) {
        // ESC key to close all dropdowns
        if (e.key === 'Escape') {
            closeAllDropdowns();
        }

        // Tab navigation
        if (e.key === 'Tab') {
            // Handle focus management
            manageFocus(e);
        }
    });
}

function initScrollEffects() {
    let lastScrollTop = 0;
    let isScrollingDown = false;
    const header = document.querySelector('.header-desktop');
    const scrollThreshold = 50; // Minimum scroll distance to trigger hide/show

    if (header) {
        window.addEventListener('scroll', function() {
            const scrollTop = window.pageYOffset || document.documentElement.scrollTop;

            // Add scrolled class for styling
            if (scrollTop > 50) {
                header.classList.add('scrolled');
            } else {
                header.classList.remove('scrolled');
            }

            // Determine scroll direction
            if (scrollTop > lastScrollTop && scrollTop > scrollThreshold) {
                // Scrolling down
                if (!isScrollingDown) {
                    header.classList.add('hide');
                    header.classList.remove('show');
                    isScrollingDown = true;
                }
            } else if (scrollTop < lastScrollTop) {
                // Scrolling up
                if (isScrollingDown) {
                    header.classList.add('show');
                    header.classList.remove('hide');
                    isScrollingDown = false;
                }
            }

            lastScrollTop = scrollTop;
        });
    }
}

function closeAllDropdowns() {
    // Close all dropdowns
    document.querySelectorAll('.dropdown-content-desktop, .notification-dropdown-desktop, .user-dropdown-menu-desktop').forEach(dropdown => {
        if (dropdown && dropdown.classList) {
            dropdown.classList.remove('show');
        }
    });
}

function manageFocus(e) {
    // Focus management for accessibility
    const focusableElements = document.querySelectorAll(
        '.nav-item-desktop, .notification-btn-desktop, .user-dropdown-trigger-desktop, .dropdown-item-desktop, .notification-item-desktop, .dropdown-menu-item-desktop'
    );

    if (focusableElements.length > 0) {
        const firstElement = focusableElements[0];
        const lastElement = focusableElements[focusableElements.length - 1];

        if (e.shiftKey) {
            // Shift + Tab
            if (document.activeElement === firstElement) {
                if (lastElement && lastElement.focus) {
                    lastElement.focus();
                    e.preventDefault();
                }
            }
        } else {
            // Tab
            if (document.activeElement === lastElement) {
                if (firstElement && firstElement.focus) {
                    firstElement.focus();
                    e.preventDefault();
                }
            }
        }
    }
}

function loadNotificationData() {
    // Removed simulated data - notifications are now handled by Livewire
    // Real notification data is loaded via Livewire component
}

function updateNotificationBadge(badgeId, count) {
    const badge = document.getElementById(badgeId);
    if (badge) {
        if (count > 0) {
            badge.textContent = count > 99 ? '99+' : count;
            badge.style.display = 'flex';
        } else {
            badge.style.display = 'none';
        }
    }
}

function loadNotificationItems() {
    // Removed - notification items are now handled by Livewire component
}

function createNotificationItem(notification) {
    // Removed - notification items are now handled by Livewire component
}

// Utility functions
function debounce(func, wait) {
    let timeout;
    return function executedFunction(...args) {
        const later = () => {
            clearTimeout(timeout);
            func(...args);
        };
        clearTimeout(timeout);
        timeout = setTimeout(later, wait);
    };
}

// Export functions for potential use in other scripts
window.HeaderDesktop = {
    closeAllDropdowns,
    updateNotificationBadge,
    loadNotificationData
};

function initMenuInteractions() {
    // Add click handlers for menu items
    const menuItems = document.querySelectorAll('.dropdown-menu-item-desktop');
    menuItems.forEach(item => {
        if (item) {
            item.addEventListener('click', function(e) {
                // Remove active class from all items
                menuItems.forEach(other => {
                    if (other && other.classList) {
                        other.classList.remove('active');
                    }
                });
                // Add active class to clicked item
                if (this.classList) {
                    this.classList.add('active');

                    // Add click effect
                    this.style.transform = 'scale(0.95)';
                    setTimeout(() => {
                        this.style.transform = 'scale(1)';
                    }, 150);
                }
            });
        }
    });

    // Add hover effects for notification items
    const notificationItems = document.querySelectorAll('.notification-item-desktop');
    notificationItems.forEach(item => {
        if (item) {
            item.addEventListener('click', function() {
                // Mark as read (visual feedback)
                this.style.opacity = '0.7';
                setTimeout(() => {
                    this.style.opacity = '1';
                }, 200);
            });
        }
    });

    // Handle nav item clicks for active state
    const navItems = document.querySelectorAll('.nav-item-desktop');
    navItems.forEach(item => {
        if (item) {
            item.addEventListener('click', function(e) {
                // Remove active from all nav items
                navItems.forEach(other => {
                    if (other && other.classList) {
                        other.classList.remove('active');
                    }
                });
                // Add active to clicked nav item
                if (this.classList) {
                    this.classList.add('active');
                }
            });
        }
    });

    // Handle dropdown item clicks
    const dropdownItems = document.querySelectorAll('.dropdown-item-desktop');
    dropdownItems.forEach(item => {
        if (item) {
            item.addEventListener('click', function(e) {
                // Remove active from all dropdown items
                dropdownItems.forEach(other => {
                    if (other && other.classList) {
                        other.classList.remove('active');
                    }
                });
                // Add active to clicked dropdown item
                if (this.classList) {
                    this.classList.add('active');
                }
            });
        }
    });
}
