// Game Info Video Error Handling
document.addEventListener('DOMContentLoaded', function() {

    // Handle iframe load errors
    function handleVideoError(iframe) {
        const container = iframe.closest('.game-content-video-container');
        const fallback = container.querySelector('.video-fallback');

        if (fallback) {
            // Hide iframe and show fallback
            iframe.style.display = 'none';
            fallback.style.display = 'flex';
            fallback.classList.add('show');
        }
    }

    // Monitor all YouTube iframes
    function monitorYouTubeIframes() {
        const iframes = document.querySelectorAll('.game-content-video');

        iframes.forEach(iframe => {
            // Add loading indicator
            const container = iframe.closest('.game-content-video-container');
            const loadingDiv = document.createElement('div');
            loadingDiv.className = 'video-loading';
            loadingDiv.innerHTML = `
                <div class="text-center">
                    <div class="video-loading-spinner"></div>
                    <p class="text-white mt-4">Loading video...</p>
                </div>
            `;
            container.appendChild(loadingDiv);

            // Handle successful load
            iframe.addEventListener('load', function() {
                setTimeout(() => {
                    loadingDiv.remove();
                }, 1000);

                // Check if iframe content is actually loaded
                setTimeout(() => {
                    try {
                        const iframeDoc = iframe.contentDocument || iframe.contentWindow.document;
                        if (!iframeDoc || iframeDoc.title.includes('error') || iframeDoc.title.includes('unavailable')) {
                            handleVideoError(iframe);
                        }
                    } catch (e) {
                        // Cross-origin restrictions prevent checking, assume it's working
                        // Video loaded (cross-origin)
                    }
                }, 2000);
            });

            // Handle error
            iframe.addEventListener('error', function() {
                loadingDiv.remove();
                handleVideoError(iframe);
            });

            // Fallback timeout for slow connections
            setTimeout(() => {
                if (loadingDiv.parentNode) {
                    loadingDiv.remove();
                    // If still no response after 10 seconds, show fallback
                    const fallback = container.querySelector('.video-fallback');
                    if (fallback && iframe.style.display !== 'none') {
                        // This might be a blocked video, show option to open externally
                        const retryBtn = document.createElement('button');
                        retryBtn.innerHTML = '<i class="fas fa-redo mr-2"></i>Retry';
                        retryBtn.className = 'ml-2 px-3 py-1 bg-blue-600 hover:bg-blue-500 text-white rounded text-sm transition-colors';
                        retryBtn.onclick = () => {
                            iframe.src = iframe.src; // Reload iframe
                        };

                        const fallbackContent = fallback.querySelector('div');
                        if (fallbackContent) {
                            fallbackContent.appendChild(retryBtn);
                        }
                    }
                }
            }, 10000);
        });
    }

    // Initialize monitoring
    monitorYouTubeIframes();

    // Re-monitor when new content is loaded (Livewire updates)
    document.addEventListener('livewire:morph.updated', function() {
        setTimeout(monitorYouTubeIframes, 500);
    });

    // Also monitor for Livewire navigations
    if (window.Livewire) {
        Livewire.on('contentUpdated', function() {
            setTimeout(monitorYouTubeIframes, 500);
        });
    }
});

// Test function to check if YouTube is accessible
function testYouTubeConnectivity() {
    const testImg = new Image();
    testImg.onload = () => { /* YouTube is accessible */ };
    testImg.onerror = () => { /* YouTube may be blocked or slow */ };
    testImg.src = 'https://img.youtube.com/vi/dQw4w9WgXcQ/0.jpg';
}

// Run connectivity test
testYouTubeConnectivity();
