/**
 * Event Rankings JavaScript Module
 * Handles countdown timer, live updates, and interactive features
 */

class EventRankings {
    constructor() {
        this.countdownInterval = null;
        this.liveIndicator = null;
        this.isInitialized = false;
        this.animationQueue = [];
        this.debounceTimer = null;
        this.isDestroyed = false;
        this.activeTimeouts = []; // Track timeouts for cleanup
        this.observers = []; // Track observers for cleanup

        // Ranking tracking
        this.previousRankings = new Map();
        this.rankingHistory = [];
        this.animationInProgress = false;

        this.init();
    }

    /**
     * Track ranking changes and apply animations
     */
    trackRankingChanges() {
        if (this.animationInProgress) return;

        this.animationInProgress = true;

        // Get current rankings
        const currentRankings = this.getCurrentRankings();

        // Load previous rankings from localStorage if not available
        if (this.previousRankings.size === 0) {
            this.loadPreviousRankings();
        }

        // Compare with previous rankings
        const changes = this.compareRankings(currentRankings);

        // Apply animations based on changes
        this.applyRankingAnimations(changes);

        // Store current rankings for next comparison
        this.previousRankings = currentRankings;
        this.savePreviousRankings();

        // Reset animation flag after animations complete
        setTimeout(() => {
            this.animationInProgress = false;
        }, 2000);
    }

    /**
     * Get current rankings from DOM
     */
    getCurrentRankings() {
        const rankings = new Map();

        // Get podium players
        const podiumPlayers = document.querySelectorAll('.podium-player[data-player-id]');
        podiumPlayers.forEach(element => {
            const playerId = element.dataset.playerId;
            const rank = parseInt(element.dataset.rank);
            const points = parseInt(element.dataset.points);

            rankings.set(playerId, {
                rank,
                points,
                element,
                type: 'podium'
            });
        });

        // Get table players
        const tableRows = document.querySelectorAll('.table-row[data-player-id]');
        tableRows.forEach(element => {
            const playerId = element.dataset.playerId;
            const rank = parseInt(element.dataset.rank);
            const points = parseInt(element.dataset.points);

            rankings.set(playerId, {
                rank,
                points,
                element,
                type: 'table'
            });
        });

        return rankings;
    }

    /**
     * Compare current rankings with previous rankings
     */
    compareRankings(currentRankings) {
        const changes = [];

        // Check for rank changes
        currentRankings.forEach((current, playerId) => {
            const previous = this.previousRankings.get(playerId);

            if (!previous) {
                // New player
                changes.push({
                    playerId,
                    type: 'new',
                    current,
                    element: current.element
                });
            } else if (previous.rank !== current.rank) {
                // Rank change
                const changeType = current.rank < previous.rank ? 'up' : 'down';
                changes.push({
                    playerId,
                    type: 'rank_change',
                    direction: changeType,
                    previous,
                    current,
                    element: current.element,
                    rankDiff: Math.abs(current.rank - previous.rank)
                });
            } else if (previous.points !== current.points) {
                // Points change
                changes.push({
                    playerId,
                    type: 'points_change',
                    previous,
                    current,
                    element: current.element,
                    pointsDiff: current.points - previous.points
                });
            }
        });

        // Check for players who left rankings
        this.previousRankings.forEach((previous, playerId) => {
            if (!currentRankings.has(playerId)) {
                changes.push({
                    playerId,
                    type: 'removed',
                    previous
                });
            }
        });

        return changes;
    }

    /**
     * Apply animations based on ranking changes
     */
    applyRankingAnimations(changes) {
        changes.forEach((change, index) => {
            // Delay animations slightly for better visual effect
            const timeoutId = setTimeout(() => {
                if (!this.isDestroyed) {
                    this.applyChangeAnimation(change);
                }
            }, index * 200);

            this.activeTimeouts.push(timeoutId);
        });
    }

    /**
     * Apply specific animation for a change
     */
    applyChangeAnimation(change) {
        const { element, type, direction } = change;

        if (!element) return;

        // Remove existing animation classes
        this.removeAnimationClasses(element);

        switch (type) {
            case 'new':
                this.animateNewEntry(element);
                break;

            case 'rank_change':
                if (direction === 'up') {
                    this.animateRankUp(element, change.rankDiff);
                } else {
                    this.animateRankDown(element, change.rankDiff);
                }
                break;

            case 'points_change':
                this.animatePointsChange(element, change.pointsDiff);
                break;
        }

        // Add highlight effect
        this.addHighlightEffect(element);
    }

    /**
     * Animate new entry
     */
    animateNewEntry(element) {
        element.classList.add('new-entry-animation');

        // Show notification for new entry
        const playerId = element.dataset.playerId;
        this.showNotification(`${playerId} joined the rankings!`, 'info');

        // Remove animation class after completion
        const timeoutId = setTimeout(() => {
            if (!this.isDestroyed) {
                element.classList.remove('new-entry-animation');
            }
        }, 2000);

        this.activeTimeouts.push(timeoutId);
    }

    /**
     * Animate rank up
     */
    animateRankUp(element, rankDiff) {
        element.classList.add('rank-up-animation');

        // Add special effects for significant rank improvements
        if (rankDiff >= 5) {
            this.addSparkleEffect(element);
        }

        // Show notification
        const playerId = element.dataset.playerId;
        this.showNotification(`🚀 ${playerId} moved up ${rankDiff} rank${rankDiff > 1 ? 's' : ''}!`, 'success');

        // Remove animation class after completion
        const timeoutId = setTimeout(() => {
            if (!this.isDestroyed) {
                element.classList.remove('rank-up-animation');
            }
        }, 1500);

        this.activeTimeouts.push(timeoutId);
    }

    /**
     * Animate rank down
     */
    animateRankDown(element, rankDiff) {
        element.classList.add('rank-down-animation');

        // Show notification
        const playerId = element.dataset.playerId;
        this.showNotification(`📉 ${playerId} moved down ${rankDiff} rank${rankDiff > 1 ? 's' : ''}`, 'warning');

        // Remove animation class after completion
        setTimeout(() => {
            element.classList.remove('rank-down-animation');
        }, 1500);
    }

    /**
     * Animate points change
     */
    animatePointsChange(element, pointsDiff) {
        const pointsElement = element.querySelector('.player-points, .points-cell');

        if (pointsElement) {
            pointsElement.classList.add('points-increase-animation');

            if (pointsDiff > 0) {
                // Show floating points indicator
                this.showFloatingPoints(element, pointsDiff);
            }

            setTimeout(() => {
                pointsElement.classList.remove('points-increase-animation');
            }, 1000);
        }
    }

    /**
     * Add highlight effect
     */
    addHighlightEffect(element) {
        element.classList.add('ranking-highlight');

        setTimeout(() => {
            element.classList.remove('ranking-highlight');
        }, 1500);
    }

    /**
     * Add sparkle effect for major improvements
     */
    addSparkleEffect(element) {
        const sparkles = document.createElement('div');
        sparkles.className = 'sparkle-effect';
        sparkles.innerHTML = '✨✨✨';
        sparkles.style.cssText = `
            position: absolute;
            top: -20px;
            left: 50%;
            transform: translateX(-50%);
            color: #ffd700;
            font-size: 1.5rem;
            z-index: 1000;
            animation: sparkle 2s ease-out forwards;
            pointer-events: none;
        `;

        element.style.position = 'relative';
        element.appendChild(sparkles);

        setTimeout(() => {
            if (sparkles.parentNode) {
                sparkles.parentNode.removeChild(sparkles);
            }
        }, 2000);
    }

    /**
     * Show floating points indicator
     */
    showFloatingPoints(element, points) {
        const floatingPoints = document.createElement('div');
        floatingPoints.className = 'floating-points';
        floatingPoints.textContent = `+${points}`;
        floatingPoints.style.cssText = `
            position: absolute;
            top: -10px;
            right: -10px;
            color: #22c55e;
            font-weight: bold;
            font-size: 0.875rem;
            z-index: 1000;
            animation: floatUp 2s ease-out forwards;
            pointer-events: none;
        `;

        element.style.position = 'relative';
        element.appendChild(floatingPoints);

        setTimeout(() => {
            if (floatingPoints.parentNode) {
                floatingPoints.parentNode.removeChild(floatingPoints);
            }
        }, 2000);
    }

    /**
     * Remove animation classes
     */
    removeAnimationClasses(element) {
        const animationClasses = [
            'new-entry-animation',
            'rank-up-animation',
            'rank-down-animation',
            'points-increase-animation',
            'ranking-highlight'
        ];

        element.classList.remove(...animationClasses);
    }

    /**
     * Save previous rankings to localStorage
     */
    savePreviousRankings() {
        try {
            const rankingsData = Array.from(this.previousRankings.entries()).map(([playerId, data]) => [
                playerId,
                {
                    rank: data.rank,
                    points: data.points,
                    type: data.type
                }
            ]);
            localStorage.setItem('eventRankings_previous', JSON.stringify(rankingsData));
        } catch (error) {
            // Failed to save rankings to localStorage
        }
    }

    /**
     * Load previous rankings from localStorage
     */
    loadPreviousRankings() {
        try {
            const savedData = localStorage.getItem('eventRankings_previous');
            if (savedData) {
                const rankingsData = JSON.parse(savedData);
                this.previousRankings = new Map(rankingsData.map(([playerId, data]) => [
                    playerId,
                    data
                ]));
            }
        } catch (error) {
            // Failed to load rankings from localStorage
            this.previousRankings = new Map();
        }
    }

    /**
     * Initialize the Event Rankings functionality
     */
    init() {
        if (this.isInitialized) return;

        document.addEventListener('DOMContentLoaded', () => {
            this.initializeCountdown();
            this.initializeLiveIndicator();
            this.initializeTableAnimations();
            this.initializeLivewireEvents();
            this.initializeErrorHandling();
            this.initializePerformanceOptimizations();

            this.isInitialized = true;
            // Event Rankings initialized silently
        });
    }

    /**
     * Initialize countdown timer functionality
     */
    initializeCountdown() {
        const countdownElement = document.getElementById('countdown-timer');
        if (!countdownElement) return;

        const endTime = parseInt(countdownElement.dataset.endTime);
        if (!endTime || isNaN(endTime)) {
            // Invalid end time for countdown
            return;
        }

        this.startCountdown(endTime);
    }

    /**
     * Start the countdown timer
     * @param {number} endTime - End time in milliseconds
     */
    startCountdown(endTime) {
        // Clear any existing interval
        if (this.countdownInterval) {
            clearInterval(this.countdownInterval);
            this.countdownInterval = null;
        }

        const updateCountdown = () => {
            // Check if component is destroyed
            if (this.isDestroyed) {
                this.handleCountdownEnd();
                return;
            }

            try {
                const now = new Date().getTime();
                const distance = endTime - now;

                if (distance < 0) {
                    this.handleCountdownEnd();
                    return;
                }

                const time = this.calculateTimeUnits(distance);
                this.updateCountdownDisplay(time);
            } catch (error) {
                // Error updating countdown
                this.handleCountdownEnd();
            }
        };

        // Initial update
        updateCountdown();

        // Update every 1 second instead of 100ms for better performance
        this.countdownInterval = setInterval(updateCountdown, 1000);
    }

    /**
     * Calculate time units from distance
     * @param {number} distance - Time distance in milliseconds
     * @returns {Object} Time units object
     */
    calculateTimeUnits(distance) {
        return {
            days: Math.floor(distance / (1000 * 60 * 60 * 24)),
            hours: Math.floor((distance % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60)),
            minutes: Math.floor((distance % (1000 * 60 * 60)) / (1000 * 60)),
            seconds: Math.floor((distance % (1000 * 60)) / 1000)
        };
    }

    /**
     * Update countdown display elements
     * @param {Object} time - Time units object
     */
    updateCountdownDisplay(time) {
        const elements = {
            days: document.getElementById('days'),
            hours: document.getElementById('hours'),
            minutes: document.getElementById('minutes'),
            seconds: document.getElementById('seconds')
        };

        Object.keys(elements).forEach(unit => {
            const element = elements[unit];
            if (element) {
                const value = time[unit].toString().padStart(2, '0');
                if (element.textContent !== value) {
                    element.textContent = value;
                    this.animateCountdownUnit(element);
                }
            }
        });
    }

    /**
     * Animate countdown unit when value changes
     * @param {HTMLElement} element - The element to animate
     */
    animateCountdownUnit(element) {
        element.style.transform = 'scale(1.1)';
        element.style.textShadow = '0 0 20px rgba(239, 68, 68, 0.9)';

        setTimeout(() => {
            element.style.transform = 'scale(1)';
            element.style.textShadow = '0 0 10px rgba(239, 68, 68, 0.5)';
        }, 200);
    }

    /**
     * Handle countdown end
     */
    handleCountdownEnd() {
        if (this.countdownInterval) {
            clearInterval(this.countdownInterval);
            this.countdownInterval = null;
        }

        // Set all countdown values to 00
        ['hours', 'minutes', 'seconds', 'milliseconds'].forEach(unit => {
            const element = document.getElementById(unit);
            if (element) {
                element.textContent = '00';
            }
        });

        // Trigger event ended notification
        this.showEventEndedNotification();
    }

    /**
     * Initialize live indicator
     */
    initializeLiveIndicator() {
        this.liveIndicator = document.getElementById('live-indicator');
        if (this.liveIndicator) {
            this.updateLiveIndicatorStatus(true);
        }
    }

    /**
     * Update live indicator status
     * @param {boolean} isLive - Whether the system is live
     */
    updateLiveIndicatorStatus(isLive) {
        if (!this.liveIndicator) return;

        const dot = this.liveIndicator.querySelector('.live-dot');
        const text = this.liveIndicator.querySelector('.live-text');

        if (isLive) {
            this.liveIndicator.classList.remove('opacity-50');
            if (dot) dot.style.backgroundColor = '#22c55e';
            if (text) text.textContent = 'Live Updates';
        } else {
            this.liveIndicator.classList.add('opacity-50');
            if (dot) dot.style.backgroundColor = '#ef4444';
            if (text) text.textContent = 'Connection Lost';
        }
    }

    /**
     * Initialize table animations
     */
    initializeTableAnimations() {
        const tableRows = document.querySelectorAll('.table-row');

        tableRows.forEach((row, index) => {
            // Stagger initial animation
            row.style.opacity = '0';
            row.style.transform = 'translateY(20px)';

            setTimeout(() => {
                row.style.transition = 'all 0.5s ease';
                row.style.opacity = '1';
                row.style.transform = 'translateY(0)';
            }, index * 100);

            // Add hover effects
            row.addEventListener('mouseenter', () => {
                this.animateTableRow(row, 'enter');
            });

            row.addEventListener('mouseleave', () => {
                this.animateTableRow(row, 'leave');
            });
        });
    }

    /**
     * Animate table row on hover
     * @param {HTMLElement} row - The table row element
     * @param {string} action - 'enter' or 'leave'
     */
    animateTableRow(row, action) {
        if (action === 'enter') {
            row.style.transform = 'translateX(5px) scale(1.02)';
            row.style.backgroundColor = 'rgba(168, 85, 247, 0.15)';
            row.style.boxShadow = '0 4px 15px rgba(168, 85, 247, 0.2)';
        } else {
            row.style.transform = 'translateX(0) scale(1)';
            row.style.backgroundColor = 'transparent';
            row.style.boxShadow = 'none';
        }
    }

    /**
     * Initialize Livewire event listeners
     */
    initializeLivewireEvents() {
        if (typeof Livewire === 'undefined') {
            // Livewire not found, event listeners not initialized
            return;
        }

        document.addEventListener('livewire:initialized', () => {
            // Livewire initialized, setting up event listeners

            // Rankings updated event
            Livewire.on('rankings-updated', () => {
                this.handleRankingsUpdate();
            });

            // Top players loaded event
            Livewire.on('top-players-loaded', () => {
                this.handleRankingsUpdate();
            });

            // Other players loaded event
            Livewire.on('other-players-loaded', () => {
                this.handleRankingsUpdate();
            });

            // Events updated event
            Livewire.on('events-updated', () => {
                this.handleEventsUpdate();
            });

            // Timer updated event
            Livewire.on('timer-updated', () => {
                this.handleTimerUpdate();
            });

            // Page refreshed event
            Livewire.on('page-refreshed', () => {
                this.handlePageRefresh();
            });
        });
    }

    /**
     * Handle rankings update
     */
    handleRankingsUpdate() {
        // Preserve translations
        this.preserveTranslations();

        // Track ranking changes and animate
        this.trackRankingChanges();

        // Add visual feedback for ranking updates
        this.animateLiveIndicator();

        // Update table animations
        setTimeout(() => {
            this.initializeTableAnimations();
        }, 300);
    }

    /**
     * Handle events update
     */
    handleEventsUpdate() {
        // Preserve translations
        this.preserveTranslations();

        // Animate event tabs
        const tabs = document.querySelectorAll('button[wire\\:click*="selectEvent"]');
        tabs.forEach(tab => {
            tab.style.transform = 'scale(1.02)';
            setTimeout(() => {
                tab.style.transform = 'scale(1)';
            }, 200);
        });
    }

    /**
     * Handle timer update
     */
    handleTimerUpdate() {
        // Preserve translations
        this.preserveTranslations();

        // Reinitialize countdown with new data
        setTimeout(() => {
            this.initializeCountdown();
        }, 100);

        // Add timer glow effect
        const timer = document.getElementById('countdown-timer');
        if (timer) {
            timer.style.textShadow = '0 0 15px rgba(239, 68, 68, 0.8)';
            setTimeout(() => {
                timer.style.textShadow = '0 0 10px rgba(239, 68, 68, 0.5)';
            }, 500);
        }
    }

    /**
     * Handle page refresh
     */
    handlePageRefresh() {
        // Preserve translations
        this.preserveTranslations();

        // Reinitialize all components
        setTimeout(() => {
            this.initializeCountdown();
            this.initializeTableAnimations();
        }, 100);

        // Show refresh feedback
        this.showRefreshNotification();
    }

    /**
     * Animate live indicator
     */
    animateLiveIndicator() {
        if (!this.liveIndicator) return;

        this.liveIndicator.classList.add('animate-pulse');
        setTimeout(() => {
            this.liveIndicator.classList.remove('animate-pulse');
        }, 1000);
    }

    /**
     * Animate ranking changes
     */
    animateRankingChanges() {
        const table = document.querySelector('tbody');
        if (table) {
            table.style.opacity = '0.7';
            table.style.transform = 'translateY(10px)';

            setTimeout(() => {
                table.style.opacity = '1';
                table.style.transform = 'translateY(0)';
            }, 300);
        }

        // Animate podium players
        const podiumPlayers = document.querySelectorAll('.podium-player');
        podiumPlayers.forEach((player, index) => {
            setTimeout(() => {
                player.style.transform = 'scale(1.05)';
                setTimeout(() => {
                    player.style.transform = 'scale(1)';
                }, 200);
            }, index * 100);
        });
    }

    /**
     * Preserve translations after Livewire updates
     */
    preserveTranslations() {
        this.debounce(() => {
            if (window.languageManager && typeof window.languageManager.debounceApplyTranslations === 'function') {
                window.languageManager.debounceApplyTranslations();
            }
        }, 100);
    }

    /**
     * Initialize error handling
     */
    initializeErrorHandling() {
        window.addEventListener('error', (event) => {
            // Event Rankings Error
            this.handleError(event.error);
        });

        // Handle Livewire errors
        document.addEventListener('livewire:request', () => {
            this.updateLiveIndicatorStatus(true);
        });

        document.addEventListener('livewire:failed', (event) => {
            // Livewire request failed
            this.updateLiveIndicatorStatus(false);
            this.showErrorNotification('Connection failed. Retrying...');
        });
    }

    /**
     * Handle errors gracefully
     * @param {Error} error - The error object
     */
    handleError(error) {
        // Stop countdown if there's an error
        if (this.countdownInterval) {
            clearInterval(this.countdownInterval);
        }

        // Update live indicator
        this.updateLiveIndicatorStatus(false);

        // Show user-friendly error message
        this.showErrorNotification('Something went wrong. Please refresh the page.');
    }

    /**
     * Initialize performance optimizations
     */
    initializePerformanceOptimizations() {
        // Lazy load avatar images
        this.initializeLazyLoading();

        // Optimize scroll performance
        this.initializeScrollOptimization();

        // Preload critical resources
        this.preloadCriticalResources();
    }

    /**
     * Initialize lazy loading for avatar images
     */
    initializeLazyLoading() {
        if ('IntersectionObserver' in window) {
            const imageObserver = new IntersectionObserver((entries) => {
                entries.forEach(entry => {
                    if (entry.isIntersecting) {
                        const img = entry.target;
                        if (img.dataset.src) {
                            img.src = img.dataset.src;
                            img.removeAttribute('data-src');
                            imageObserver.unobserve(img);
                        }
                    }
                });
            });

            // Store observer for cleanup
            this.observers.push(imageObserver);

            document.querySelectorAll('img[data-src]').forEach(img => {
                imageObserver.observe(img);
            });
        }
    }

    /**
     * Initialize scroll performance optimization
     */
    initializeScrollOptimization() {
        let ticking = false;

        const handleScroll = () => {
            if (!ticking) {
                requestAnimationFrame(() => {
                    // Add scroll-based animations here if needed
                    ticking = false;
                });
                ticking = true;
            }
        };

        window.addEventListener('scroll', handleScroll, { passive: true });
    }

    /**
     * Preload critical resources
     */
    preloadCriticalResources() {
        // Preload avatar images for top players
        const topPlayerAvatars = document.querySelectorAll('.player-avatar, .player-small-avatar');
        topPlayerAvatars.forEach(img => {
            if (img.src) {
                const preloadImg = new Image();
                preloadImg.src = img.src;
            }
        });
    }

    /**
     * Show event ended notification
     */
    showEventEndedNotification() {
        this.showNotification('Event has ended!', 'info');
    }

    /**
     * Show refresh notification
     */
    showRefreshNotification() {
        this.showNotification('Rankings updated', 'success');
    }

    /**
     * Show error notification
     * @param {string} message - Error message
     */
    showErrorNotification(message) {
        this.showNotification(message, 'error');
    }

    /**
     * Show notification
     * @param {string} message - Notification message
     * @param {string} type - Notification type (success, error, info)
     */
    showNotification(message, type = 'info') {
        // Create notification element
        const notification = document.createElement('div');
        notification.className = `notification notification-${type}`;
        notification.innerHTML = `
            <div class="notification-content">
                <span class="notification-message">${message}</span>
                <button class="notification-close">&times;</button>
            </div>
        `;

        // Add styles
        notification.style.cssText = `
            position: fixed;
            top: 20px;
            right: 20px;
            background: ${type === 'error' ? '#ef4444' : type === 'success' ? '#22c55e' : '#3b82f6'};
            color: white;
            padding: 1rem 1.5rem;
            border-radius: 0.5rem;
            box-shadow: 0 10px 25px rgba(0, 0, 0, 0.3);
            z-index: 1000;
            animation: slideIn 0.3s ease;
            backdrop-filter: blur(10px);
        `;

        // Add to DOM
        document.body.appendChild(notification);

        // Auto remove after 5 seconds
        setTimeout(() => {
            this.removeNotification(notification);
        }, 5000);

        // Manual close
        const closeBtn = notification.querySelector('.notification-close');
        if (closeBtn) {
            closeBtn.addEventListener('click', () => {
                this.removeNotification(notification);
            });
        }
    }

    /**
     * Remove notification
     * @param {HTMLElement} notification - Notification element
     */
    removeNotification(notification) {
        notification.style.animation = 'slideOut 0.3s ease';
        setTimeout(() => {
            if (notification.parentNode) {
                notification.parentNode.removeChild(notification);
            }
        }, 300);
    }

    /**
     * Debounce function
     * @param {Function} func - Function to debounce
     * @param {number} wait - Wait time in milliseconds
     */
    debounce(func, wait) {
        clearTimeout(this.debounceTimer);
        this.debounceTimer = setTimeout(func, wait);
    }

    /**
     * Cleanup function
     */
    destroy() {
        this.isDestroyed = true;

        // Clear countdown interval
        if (this.countdownInterval) {
            clearInterval(this.countdownInterval);
            this.countdownInterval = null;
        }

        // Clear debounce timer
        if (this.debounceTimer) {
            clearTimeout(this.debounceTimer);
            this.debounceTimer = null;
        }

        // Clear all active timeouts
        this.activeTimeouts.forEach(timeoutId => {
            clearTimeout(timeoutId);
        });
        this.activeTimeouts = [];

        // Disconnect all observers
        this.observers.forEach(observer => {
            observer.disconnect();
        });
        this.observers = [];

        // Clear localStorage data if needed
        try {
            localStorage.removeItem('eventRankings_previous');
        } catch (error) {
            // Failed to clear localStorage
        }

        // Clear rankings data
        this.previousRankings.clear();
        this.rankingHistory = [];

        this.isInitialized = false;
        // Event Rankings destroyed
    }
}

// CSS animations for notifications
const notificationStyles = document.createElement('style');
notificationStyles.textContent = `
    @keyframes slideIn {
        from {
            transform: translateX(100%);
            opacity: 0;
        }
        to {
            transform: translateX(0);
            opacity: 1;
        }
    }

    @keyframes slideOut {
        from {
            transform: translateX(0);
            opacity: 1;
        }
        to {
            transform: translateX(100%);
            opacity: 0;
        }
    }

    .notification-content {
        display: flex;
        align-items: center;
        justify-content: space-between;
        gap: 1rem;
    }

    .notification-close {
        background: none;
        border: none;
        color: white;
        font-size: 1.25rem;
        cursor: pointer;
        padding: 0;
        width: 24px;
        height: 24px;
        display: flex;
        align-items: center;
        justify-content: center;
        border-radius: 50%;
        transition: background-color 0.2s ease;
    }

    .notification-close:hover {
        background: rgba(255, 255, 255, 0.2);
    }
`;
document.head.appendChild(notificationStyles);

// Additional CSS for ranking animations
const rankingAnimationStyles = document.createElement('style');
rankingAnimationStyles.textContent = `
    @keyframes sparkle {
        0% {
            opacity: 0;
            transform: translateX(-50%) translateY(0) scale(0.5) rotate(0deg);
        }
        25% {
            opacity: 1;
            transform: translateX(-50%) translateY(-5px) scale(0.8) rotate(90deg);
        }
        50% {
            opacity: 1;
            transform: translateX(-50%) translateY(-10px) scale(1) rotate(180deg);
        }
        75% {
            opacity: 0.8;
            transform: translateX(-50%) translateY(-15px) scale(1.1) rotate(270deg);
        }
        100% {
            opacity: 0;
            transform: translateX(-50%) translateY(-20px) scale(0.5) rotate(360deg);
        }
    }

    @keyframes floatUp {
        0% {
            opacity: 1;
            transform: translateY(0) scale(1);
        }
        50% {
            opacity: 0.8;
            transform: translateY(-15px) scale(1.1);
        }
        100% {
            opacity: 0;
            transform: translateY(-30px) scale(1.2);
        }
    }
`;
document.head.appendChild(rankingAnimationStyles);

// Initialize the Event Rankings system
window.eventRankings = new EventRankings();

// Cleanup on page unload
window.addEventListener('beforeunload', () => {
    if (window.eventRankings) {
        window.eventRankings.destroy();
    }
});

// Cleanup on visibility change for better performance
document.addEventListener('visibilitychange', () => {
    if (document.hidden && window.eventRankings) {
        // Pause animations when page is not visible
        window.eventRankings.animationInProgress = false;
    }
});

// Export for module systems
if (typeof module !== 'undefined' && module.exports) {
    module.exports = EventRankings;
}
