/**
 * Enhanced Event Rankings JavaScript Module
 * Handles countdown timer, live updates, ranking animations, and player ownership highlighting
 */

class EventRankings {
    constructor() {
        this.countdownInterval = null;
        this.liveIndicator = null;
        this.isInitialized = false;
        this.animationQueue = [];
        this.debounceTimer = null;

        // Ranking tracking
        this.previousRankings = new Map();
        this.rankingHistory = [];
        this.animationInProgress = false;

        this.init();
    }

    /**
     * Initialize the Event Rankings functionality
     */
    init() {
        if (this.isInitialized) return;

        document.addEventListener('DOMContentLoaded', () => {
            this.initializeCountdown();
            this.initializeLiveIndicator();
            this.initializeTableAnimations();
            this.initializeLivewireEvents();
            this.initializeErrorHandling();
            this.initializePerformanceOptimizations();
            this.isInitialized = true;
        });
    }

    /**
     * Track ranking changes and apply animations
     */
    trackRankingChanges() {
        if (this.animationInProgress) return;

        this.animationInProgress = true;

        // Get current rankings
        const currentRankings = this.getCurrentRankings();

        // Compare with previous rankings
        const changes = this.compareRankings(currentRankings);

        // Apply animations based on changes
        this.applyRankingAnimations(changes);

        // Store current rankings for next comparison
        this.previousRankings = currentRankings;

        // Reset animation flag after animations complete
        setTimeout(() => {
            this.animationInProgress = false;
        }, 2000);
    }

    /**
     * Get current rankings from DOM
     */
    getCurrentRankings() {
        const rankings = new Map();

        // Get podium players
        const podiumPlayers = document.querySelectorAll('.podium-player[data-player-id]');
        podiumPlayers.forEach(element => {
            const playerId = element.dataset.playerId;
            const rank = parseInt(element.dataset.rank);
            const points = parseInt(element.dataset.points);

            rankings.set(playerId, {
                rank,
                points,
                element,
                type: 'podium'
            });
        });

        // Get table players
        const tableRows = document.querySelectorAll('.table-row[data-player-id]');
        tableRows.forEach(element => {
            const playerId = element.dataset.playerId;
            const rank = parseInt(element.dataset.rank);
            const points = parseInt(element.dataset.points);

            rankings.set(playerId, {
                rank,
                points,
                element,
                type: 'table'
            });
        });

        return rankings;
    }

    /**
     * Compare current rankings with previous rankings
     */
    compareRankings(currentRankings) {
        const changes = [];

        // Check for rank changes
        currentRankings.forEach((current, playerId) => {
            const previous = this.previousRankings.get(playerId);

            if (!previous) {
                // New player
                changes.push({
                    playerId,
                    type: 'new',
                    current,
                    element: current.element
                });
            } else if (previous.rank !== current.rank) {
                // Rank change
                const changeType = current.rank < previous.rank ? 'up' : 'down';
                changes.push({
                    playerId,
                    type: 'rank_change',
                    direction: changeType,
                    previous,
                    current,
                    element: current.element,
                    rankDiff: Math.abs(current.rank - previous.rank)
                });
            } else if (previous.points !== current.points) {
                // Points change
                changes.push({
                    playerId,
                    type: 'points_change',
                    previous,
                    current,
                    element: current.element,
                    pointsDiff: current.points - previous.points
                });
            }
        });

        return changes;
    }

    /**
     * Apply animations based on ranking changes
     */
    applyRankingAnimations(changes) {
        changes.forEach((change, index) => {
            // Delay animations slightly for better visual effect
            setTimeout(() => {
                this.applyChangeAnimation(change);
            }, index * 200);
        });
    }

    /**
     * Apply specific animation for a change
     */
    applyChangeAnimation(change) {
        const { element, type, direction } = change;

        if (!element) return;

        // Remove existing animation classes
        this.removeAnimationClasses(element);

        switch (type) {
            case 'new':
                this.animateNewEntry(element);
                break;

            case 'rank_change':
                if (direction === 'up') {
                    this.animateRankUp(element, change.rankDiff);
                } else {
                    this.animateRankDown(element, change.rankDiff);
                }
                break;

            case 'points_change':
                this.animatePointsChange(element, change.pointsDiff);
                break;
        }

        // Add highlight effect
        this.addHighlightEffect(element);
    }

    /**
     * Animate new entry
     */
    animateNewEntry(element) {
        element.classList.add('new-entry-animation');

        // Show notification for new entry
        const playerId = element.dataset.playerId;
        this.showNotification(`${playerId} joined the rankings!`, 'info');

        // Remove animation class after completion
        setTimeout(() => {
            element.classList.remove('new-entry-animation');
        }, 2000);
    }

    /**
     * Animate rank up
     */
    animateRankUp(element, rankDiff) {
        element.classList.add('rank-up-animation');

        // Add special effects for significant rank improvements
        if (rankDiff >= 5) {
            this.addSparkleEffect(element);
        }

        // Show notification
        const playerId = element.dataset.playerId;
        this.showNotification(`🚀 ${playerId} moved up ${rankDiff} rank${rankDiff > 1 ? 's' : ''}!`, 'success');

        // Remove animation class after completion
        setTimeout(() => {
            element.classList.remove('rank-up-animation');
        }, 1500);
    }

    /**
     * Animate rank down
     */
    animateRankDown(element, rankDiff) {
        element.classList.add('rank-down-animation');

        // Show notification
        const playerId = element.dataset.playerId;
        this.showNotification(`📉 ${playerId} moved down ${rankDiff} rank${rankDiff > 1 ? 's' : ''}`, 'warning');

        // Remove animation class after completion
        setTimeout(() => {
            element.classList.remove('rank-down-animation');
        }, 1500);
    }

    /**
     * Animate points change
     */
    animatePointsChange(element, pointsDiff) {
        const pointsElement = element.querySelector('.player-points, .points-cell');

        if (pointsElement) {
            pointsElement.classList.add('points-increase-animation');

            if (pointsDiff > 0) {
                // Show floating points indicator
                this.showFloatingPoints(element, pointsDiff);
            }

            setTimeout(() => {
                pointsElement.classList.remove('points-increase-animation');
            }, 1000);
        }
    }

    /**
     * Add highlight effect
     */
    addHighlightEffect(element) {
        element.classList.add('ranking-highlight');

        setTimeout(() => {
            element.classList.remove('ranking-highlight');
        }, 1500);
    }

    /**
     * Add sparkle effect for major improvements
     */
    addSparkleEffect(element) {
        const sparkles = document.createElement('div');
        sparkles.className = 'sparkle-effect';
        sparkles.innerHTML = '✨✨✨';
        sparkles.style.cssText = `
            position: absolute;
            top: -20px;
            left: 50%;
            transform: translateX(-50%);
            color: #ffd700;
            font-size: 1.5rem;
            z-index: 1000;
            animation: sparkle 2s ease-out forwards;
            pointer-events: none;
        `;

        element.style.position = 'relative';
        element.appendChild(sparkles);

        setTimeout(() => {
            if (sparkles.parentNode) {
                sparkles.parentNode.removeChild(sparkles);
            }
        }, 2000);
    }

    /**
     * Show floating points indicator
     */
    showFloatingPoints(element, points) {
        const floatingPoints = document.createElement('div');
        floatingPoints.className = 'floating-points';
        floatingPoints.textContent = `+${points}`;
        floatingPoints.style.cssText = `
            position: absolute;
            top: -10px;
            right: -10px;
            color: #22c55e;
            font-weight: bold;
            font-size: 0.875rem;
            z-index: 1000;
            animation: floatUp 2s ease-out forwards;
            pointer-events: none;
        `;

        element.style.position = 'relative';
        element.appendChild(floatingPoints);

        setTimeout(() => {
            if (floatingPoints.parentNode) {
                floatingPoints.parentNode.removeChild(floatingPoints);
            }
        }, 2000);
    }

    /**
     * Remove animation classes
     */
    removeAnimationClasses(element) {
        const animationClasses = [
            'new-entry-animation',
            'rank-up-animation',
            'rank-down-animation',
            'points-increase-animation',
            'ranking-highlight'
        ];

        element.classList.remove(...animationClasses);
    }

    /**
     * Initialize countdown timer functionality
     */
    initializeCountdown() {
        const countdownElement = document.getElementById('countdown-timer');
        if (!countdownElement) return;

        const endTime = parseInt(countdownElement.dataset.endTime);
        if (!endTime || isNaN(endTime)) return;

        this.startCountdown(endTime);
    }

    /**
     * Start the countdown timer
     */
    startCountdown(endTime) {
        if (this.countdownInterval) {
            clearInterval(this.countdownInterval);
        }

        const updateCountdown = () => {
            const now = new Date().getTime();
            const distance = endTime - now;

            if (distance < 0) {
                this.handleCountdownEnd();
                return;
            }

            const time = this.calculateTimeUnits(distance);
            this.updateCountdownDisplay(time);
        };

        updateCountdown();
        this.countdownInterval = setInterval(updateCountdown, 100);
    }

    /**
     * Calculate time units from distance
     */
    calculateTimeUnits(distance) {
        return {
            days: Math.floor(distance / (1000 * 60 * 60 * 24)),
            hours: Math.floor((distance % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60)),
            minutes: Math.floor((distance % (1000 * 60 * 60)) / (1000 * 60)),
            seconds: Math.floor((distance % (1000 * 60)) / 1000)
        };
    }

    /**
     * Update countdown display elements
     */
    updateCountdownDisplay(time) {
        const elements = {
            days: document.getElementById('days'),
            hours: document.getElementById('hours'),
            minutes: document.getElementById('minutes'),
            seconds: document.getElementById('seconds')
        };

        Object.keys(elements).forEach(unit => {
            if (elements[unit]) {
                const newValue = String(time[unit]).padStart(2, '0');
                if (elements[unit].textContent !== newValue) {
                    elements[unit].textContent = newValue;
                    this.animateCountdownUnit(elements[unit]);
                }
            }
        });
    }

    /**
     * Animate countdown unit when value changes
     */
    animateCountdownUnit(element) {
        element.style.transform = 'scale(1.1)';
        element.style.textShadow = '0 0 20px rgba(239, 68, 68, 0.9)';

        setTimeout(() => {
            element.style.transform = 'scale(1)';
            element.style.textShadow = '0 0 10px rgba(239, 68, 68, 0.5)';
        }, 200);
    }

    /**
     * Handle countdown end
     */
    handleCountdownEnd() {
        if (this.countdownInterval) {
            clearInterval(this.countdownInterval);
            this.countdownInterval = null;
        }

        ['hours', 'minutes', 'seconds', 'milliseconds'].forEach(unit => {
            const element = document.getElementById(unit);
            if (element) element.textContent = '00';
        });

        this.showEventEndedNotification();
    }

    /**
     * Initialize live indicator
     */
    initializeLiveIndicator() {
        this.liveIndicator = document.getElementById('live-indicator');
        if (this.liveIndicator) {
            this.updateLiveIndicatorStatus(true);
        }
    }

    /**
     * Update live indicator status
     */
    updateLiveIndicatorStatus(isLive) {
        if (!this.liveIndicator) return;

        const dot = this.liveIndicator.querySelector('.live-dot');
        const text = this.liveIndicator.querySelector('.live-text');

        if (isLive) {
            dot?.classList.add('animate-pulse');
            text?.classList.remove('text-gray-400');
            text?.classList.add('text-green-400');
        } else {
            dot?.classList.remove('animate-pulse');
            text?.classList.add('text-gray-400');
            text?.classList.remove('text-green-400');
        }
    }

    /**
     * Initialize table animations
     */
    initializeTableAnimations() {
        const tableRows = document.querySelectorAll('.table-row');

        tableRows.forEach((row, index) => {
            row.style.animationDelay = `${index * 0.1}s`;

            row.addEventListener('mouseenter', () => {
                this.animateTableRow(row, 'enter');
            });

            row.addEventListener('mouseleave', () => {
                this.animateTableRow(row, 'leave');
            });
        });
    }

    /**
     * Animate table row on hover
     */
    animateTableRow(row, action) {
        if (action === 'enter') {
            row.style.transform = 'translateX(5px)';
            row.style.boxShadow = '0 4px 12px rgba(168, 85, 247, 0.3)';
        } else {
            row.style.transform = 'translateX(0)';
            row.style.boxShadow = 'none';
        }
    }

    /**
     * Initialize Livewire event listeners
     */
    initializeLivewireEvents() {
        if (typeof Livewire === 'undefined') return;

        document.addEventListener('livewire:initialized', () => {
            this.preserveTranslations();
        });

        // Listen for ranking updates
        Livewire.on('rankings-updated', () => {
            this.handleRankingsUpdate();
        });

        // Listen for events updates
        Livewire.on('events-updated', () => {
            this.handleEventsUpdate();
        });

        // Listen for timer updates
        Livewire.on('timer-updated', () => {
            this.handleTimerUpdate();
        });
    }

    /**
     * Handle rankings update
     */
    handleRankingsUpdate() {
        this.preserveTranslations();
        this.trackRankingChanges();
        this.animateLiveIndicator();

        setTimeout(() => {
            this.initializeTableAnimations();
        }, 300);
    }

    /**
     * Handle events update
     */
    handleEventsUpdate() {
        this.preserveTranslations();

        const tabs = document.querySelectorAll('button[wire\\:click*="selectEvent"]');
        tabs.forEach(tab => {
            tab.style.transform = 'scale(1.05)';
            setTimeout(() => {
                tab.style.transform = 'scale(1)';
            }, 200);
        });
    }

    /**
     * Handle timer update
     */
    handleTimerUpdate() {
        this.preserveTranslations();

        setTimeout(() => {
            this.initializeCountdown();
        }, 100);
    }

    /**
     * Animate live indicator
     */
    animateLiveIndicator() {
        if (this.liveIndicator) {
            this.liveIndicator.style.transform = 'scale(1.1)';
            setTimeout(() => {
                this.liveIndicator.style.transform = 'scale(1)';
            }, 300);
        }
    }

    /**
     * Preserve translations after Livewire updates
     */
    preserveTranslations() {
        // This function will be called to preserve any translation functionality
        // Implementation depends on your translation system
    }

    /**
     * Initialize error handling
     */
    initializeErrorHandling() {
        window.addEventListener('error', (event) => {
            this.handleError(event.error);
        });
    }

    /**
     * Handle errors gracefully
     */
    handleError(error) {
        // Event Rankings Error
        this.showErrorNotification('An error occurred while updating rankings');
    }

    /**
     * Initialize performance optimizations
     */
    initializePerformanceOptimizations() {
        this.initializeLazyLoading();
        this.initializeScrollOptimization();
        this.preloadCriticalResources();
    }

    /**
     * Initialize lazy loading for avatar images
     */
    initializeLazyLoading() {
        if ('IntersectionObserver' in window) {
            const imageObserver = new IntersectionObserver((entries, observer) => {
                entries.forEach(entry => {
                    if (entry.isIntersecting) {
                        const img = entry.target;
                        img.src = img.dataset.src;
                        img.classList.remove('lazy');
                        observer.unobserve(img);
                    }
                });
            });

            document.querySelectorAll('img[data-src]').forEach(img => {
                imageObserver.observe(img);
            });
        }
    }

    /**
     * Initialize scroll performance optimization
     */
    initializeScrollOptimization() {
        let ticking = false;

        const updateScrollEffects = () => {
            // Add scroll-based effects here
            ticking = false;
        };

        window.addEventListener('scroll', () => {
            if (!ticking) {
                requestAnimationFrame(updateScrollEffects);
                ticking = true;
            }
        });
    }

    /**
     * Preload critical resources
     */
    preloadCriticalResources() {
        const criticalImages = [
            '/assets/images/avatars/1907.gif',
            '/assets/images/avatars/default.gif'
        ];

        criticalImages.forEach(src => {
            const img = new Image();
            img.src = src;
        });
    }

    /**
     * Show event ended notification
     */
    showEventEndedNotification() {
        this.showNotification('🎯 Event has ended!', 'info');
    }

    /**
     * Show error notification
     */
    showErrorNotification(message) {
        this.showNotification(message, 'error');
    }

    /**
     * Show notification
     */
    showNotification(message, type = 'info') {
        const notification = document.createElement('div');
        notification.className = `notification notification-${type}`;
        notification.style.cssText = `
            position: fixed;
            top: 20px;
            right: 20px;
            padding: 1rem 1.5rem;
            border-radius: 0.5rem;
            color: white;
            font-weight: 600;
            z-index: 9999;
            animation: slideIn 0.3s ease-out;
            max-width: 400px;
            box-shadow: 0 10px 25px rgba(0, 0, 0, 0.3);
        `;

        // Set background color based on type
        switch (type) {
            case 'success':
                notification.style.background = 'linear-gradient(135deg, #10b981, #059669)';
                break;
            case 'error':
                notification.style.background = 'linear-gradient(135deg, #ef4444, #dc2626)';
                break;
            case 'warning':
                notification.style.background = 'linear-gradient(135deg, #f59e0b, #d97706)';
                break;
            default:
                notification.style.background = 'linear-gradient(135deg, #3b82f6, #2563eb)';
        }

        notification.innerHTML = `
            <div class="notification-content">
                <span>${message}</span>
                <button class="notification-close" onclick="this.parentNode.parentNode.remove()">×</button>
            </div>
        `;

        document.body.appendChild(notification);

        // Auto remove after 5 seconds
        setTimeout(() => {
            this.removeNotification(notification);
        }, 5000);
    }

    /**
     * Remove notification
     */
    removeNotification(notification) {
        if (notification.parentNode) {
            notification.style.animation = 'slideOut 0.3s ease-out forwards';
            setTimeout(() => {
                if (notification.parentNode) {
                    notification.parentNode.removeChild(notification);
                }
            }, 300);
        }
    }

    /**
     * Cleanup function
     */
    destroy() {
        if (this.countdownInterval) {
            clearInterval(this.countdownInterval);
        }

        if (this.debounceTimer) {
            clearTimeout(this.debounceTimer);
        }
    }
}

// Additional CSS animations for new features
const enhancedStyles = document.createElement('style');
enhancedStyles.textContent = `
    @keyframes sparkle {
        0% {
            opacity: 0;
            transform: translateX(-50%) translateY(0) scale(0.5);
        }
        50% {
            opacity: 1;
            transform: translateX(-50%) translateY(-10px) scale(1);
        }
        100% {
            opacity: 0;
            transform: translateX(-50%) translateY(-20px) scale(0.5);
        }
    }

    @keyframes floatUp {
        0% {
            opacity: 1;
            transform: translateY(0) scale(1);
        }
        100% {
            opacity: 0;
            transform: translateY(-30px) scale(1.2);
        }
    }

    .notification-content {
        display: flex;
        align-items: center;
        justify-content: space-between;
        gap: 1rem;
    }

    .notification-close {
        background: none;
        border: none;
        color: white;
        font-size: 1.25rem;
        cursor: pointer;
        padding: 0;
        width: 24px;
        height: 24px;
        display: flex;
        align-items: center;
        justify-content: center;
        border-radius: 50%;
        transition: background-color 0.2s ease;
    }

    .notification-close:hover {
        background: rgba(255, 255, 255, 0.2);
    }
`;
document.head.appendChild(enhancedStyles);

// Initialize the Enhanced Event Rankings system
window.eventRankings = new EventRankings();

// Export for module systems
if (typeof module !== 'undefined' && module.exports) {
    module.exports = EventRankings;
}
