/**
 * Gaming Cursor Effects & Animations
 * Creates interactive cursor trails, particles, and gaming effects
 */

class GamingCursorEffects {
    constructor() {
        this.isTouch = window.matchMedia('(hover: none) and (pointer: coarse)').matches;
        this.trails = [];
        this.maxTrails = 5; // Very few trails
        this.mousePos = { x: 0, y: 0 };
        this.isDestroyed = false;
        this.shouldAnimate = false;
        this.animationId = null;
        this.eventListeners = []; // Track event listeners for cleanup

        if (!this.isTouch) {
            this.init();
        }
    }    init() {
        this.createTrailElements();
        this.bindEvents();
        this.startAnimationLoop();

        // Protect hero section after DOM is ready
        this.protectHeroSection();
    }

    protectHeroSection() {
        // Use setTimeout to ensure DOM is fully loaded
        setTimeout(() => {
            // Ensure hero section maintains proper z-index and visibility
            const heroSection = document.querySelector('.hero-section, #heroSection');
            if (heroSection) {
                // Only set position if not already set
                if (!heroSection.style.position || heroSection.style.position === 'static') {
                    heroSection.style.position = 'relative';
                }
                // Don't override existing z-index unless it's auto or not set
                const currentZIndex = window.getComputedStyle(heroSection).zIndex;
                if (currentZIndex === 'auto' || !heroSection.style.zIndex) {
                    heroSection.style.zIndex = '1';
                }
            } else {
                // Hero section not found for protection
            }

            // Ensure cursor effects don't interfere with hero content
            const heroContent = document.querySelector('.hero-content');
            if (heroContent) {
                if (!heroContent.style.position || heroContent.style.position === 'static') {
                    heroContent.style.position = 'relative';
                }
                const currentZIndex = window.getComputedStyle(heroContent).zIndex;
                if (currentZIndex === 'auto' || !heroContent.style.zIndex) {
                    heroContent.style.zIndex = '5';
                }
            }
        }, 100);
    }    createTrailElements() {
        // Create very simple cursor trail elements - tiny dots
        for (let i = 0; i < this.maxTrails; i++) {
            const trail = document.createElement('div');
            trail.className = 'cursor-trail';
            trail.style.opacity = '0';
            document.body.appendChild(trail);
            this.trails.push({
                element: trail,
                x: 0,
                y: 0,
                opacity: 0
            });
        }
    }

    bindEvents() {
        // Mouse movement tracking
        const mouseMoveHandler = (e) => {
            this.updateMousePosition(e);
            this.updateCursorTrails();
        };

        const mouseDownHandler = (e) => {
            this.createClickStars(e);
        };

        document.addEventListener('mousemove', mouseMoveHandler);
        document.addEventListener('mousedown', mouseDownHandler);

        // Store for cleanup
        this.eventListeners.push(
            { element: document, event: 'mousemove', handler: mouseMoveHandler },
            { element: document, event: 'mousedown', handler: mouseDownHandler }
        );

        // Add visibility change listener to pause animations
        const visibilityHandler = () => {
            if (document.hidden) {
                this.shouldAnimate = false;
                if (this.animationId) {
                    cancelAnimationFrame(this.animationId);
                    this.animationId = null;
                }
            } else {
                this.shouldAnimate = true;
                this.startAnimationLoop();
            }
        };

        document.addEventListener('visibilitychange', visibilityHandler);
        this.eventListeners.push(
            { element: document, event: 'visibilitychange', handler: visibilityHandler }
        );
    }

    updateMousePosition(e) {
        this.mousePos = { x: e.clientX, y: e.clientY };
    }

    updateCursorTrails() {
        // Shift trails
        for (let i = this.trails.length - 1; i > 0; i--) {
            this.trails[i].x = this.trails[i - 1].x;
            this.trails[i].y = this.trails[i - 1].y;
            this.trails[i].opacity = this.trails[i - 1].opacity * 0.8;
        }

        // Update first trail
        if (this.trails.length > 0) {
            this.trails[0].x = this.mousePos.x;
            this.trails[0].y = this.mousePos.y;
            this.trails[0].opacity = 0.3;
        }

        // Apply positions
        this.trails.forEach((trail) => {
            trail.element.style.left = trail.x + 'px';
            trail.element.style.top = trail.y + 'px';
            trail.element.style.opacity = trail.opacity;
        });
    }

    createClickStars(e) {
        // Create tiny stars on click
        const starCount = 3;
        for (let i = 0; i < starCount; i++) {
            const star = document.createElement('div');
            star.className = 'click-stars';
            star.textContent = '✨';
            star.style.left = e.clientX + (Math.random() - 0.5) * 20 + 'px';
            star.style.top = e.clientY + (Math.random() - 0.5) * 20 + 'px';
            document.body.appendChild(star);

            // Remove after animation
            setTimeout(() => {
                if (star.parentNode) {
                    star.parentNode.removeChild(star);
                }
            }, 600);
        }
    }

    startAnimationLoop() {
        // Simple animation loop for trails with proper cleanup
        const animate = () => {
            // Stop animation if instance is destroyed or page is hidden
            if (this.isDestroyed || document.hidden) {
                return;
            }

            // Only animate if cursor is actually moving
            if (this.shouldAnimate) {
                this.animationId = requestAnimationFrame(animate);
            }
        };

        this.shouldAnimate = true;
        this.animationId = requestAnimationFrame(animate);
    }

    // Public methods for external control
    setTrailCount(count) {
        this.maxTrails = Math.max(1, Math.min(10, count));
    }

    toggleEffects(enabled) {
        if (enabled && !this.isTouch) {
            this.trails.forEach(trail => trail.element.style.display = 'block');
        } else {
            this.trails.forEach(trail => trail.element.style.display = 'none');
        }
    }

    // Cleanup method to prevent memory leaks
    destroy() {
        this.isDestroyed = true;
        this.shouldAnimate = false;

        // Cancel animation frame
        if (this.animationId) {
            cancelAnimationFrame(this.animationId);
            this.animationId = null;
        }

        // Remove event listeners
        this.eventListeners.forEach(({ element, event, handler }) => {
            element.removeEventListener(event, handler);
        });
        this.eventListeners = [];

        // Remove DOM elements
        this.trails.forEach(trail => {
            if (trail.element && trail.element.parentNode) {
                trail.element.parentNode.removeChild(trail.element);
            }
        });
        this.trails = [];

        // Remove any remaining click stars
        document.querySelectorAll('.click-stars').forEach(star => {
            if (star.parentNode) {
                star.parentNode.removeChild(star);
            }
        });
    }
}

// Add loading animation CSS
const loadingStyles = document.createElement('style');
loadingStyles.textContent = `
    @keyframes loading-spin {
        from { transform: rotate(0deg); }
        to { transform: rotate(360deg); }
    }
`;
document.head.appendChild(loadingStyles);

// Initialize cursor effects when ready
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', () => {
        setTimeout(() => {
            window.gamingCursorEffects = new GamingCursorEffects();
        }, 100);
    });
} else {
    // DOM is already loaded
    setTimeout(() => {
        window.gamingCursorEffects = new GamingCursorEffects();
    }, 100);
}

// Cleanup on page unload
window.addEventListener('beforeunload', () => {
    if (window.gamingCursorEffects) {
        window.gamingCursorEffects.destroy();
    }
});

// Cleanup on page visibility change for better performance
document.addEventListener('visibilitychange', () => {
    if (document.hidden && window.gamingCursorEffects) {
        window.gamingCursorEffects.shouldAnimate = false;
    }
});

// Listen for cursor loader completion
document.addEventListener('cursorsLoaded', () => {
    // Gaming cursor effects initialized
});

// Export for module systems
if (typeof module !== 'undefined' && module.exports) {
    module.exports = GamingCursorEffects;
}
