/**
 * Gaming Cursor Loader
 * Preloads all cursor files for smooth transitions
 */

class CursorLoader {
    constructor() {
        this.cursors = {
            normal: '/assets/silkroad mouse/normal cursor.cur',
            busy: '/assets/silkroad mouse/busy.cur',
            link: '/assets/silkroad mouse/link select.cur',
            text: '/assets/silkroad mouse/text.cur',
            move: '/assets/silkroad mouse/MOVE.cur',
            resize1: '/assets/silkroad mouse/RESIZE1.cur',
            resize2: '/assets/silkroad mouse/RESIZE 2.cur',
            help: '/assets/silkroad mouse/HELP.cur',
            wait: '/assets/silkroad mouse/wib.cur',
            altBusy: '/assets/silkroad mouse/ALTERNATE BUSY.cur',
            altLink: '/assets/silkroad mouse/ALTERNATE LINK SELECT.cur',
            altSelect: '/assets/silkroad mouse/ALTERNATE SELECT.cur',
            altWait: '/assets/silkroad mouse/ALTERNATE WIB.cur'
        };

        this.loaded = false;
        this.loadingPromises = [];
        this.init();
    }

    init() {
        // Preload all cursors
        this.preloadCursors();

        // Apply cursors when loaded
        this.applyWhenLoaded();
    }

    preloadCursors() {
        Object.values(this.cursors).forEach(cursorPath => {
            const promise = this.loadCursor(cursorPath);
            this.loadingPromises.push(promise);
        });
    }

    loadCursor(cursorPath) {
        return new Promise((resolve, reject) => {
            // Create a temporary element to test cursor loading
            const testElement = document.createElement('div');
            testElement.style.cursor = `url('${cursorPath}'), auto`;
            testElement.style.position = 'absolute';
            testElement.style.left = '-9999px';
            testElement.style.width = '1px';
            testElement.style.height = '1px';

            document.body.appendChild(testElement);

            // Check if cursor loaded (simplified check)
            setTimeout(() => {
                document.body.removeChild(testElement);
                resolve(cursorPath);
            }, 100);
        });
    }

    applyWhenLoaded() {
        Promise.allSettled(this.loadingPromises).then(() => {
            this.loaded = true;
            document.body.classList.add('cursors-loaded');

            // Dispatch custom event
            const event = new CustomEvent('cursorsLoaded', {
                detail: { cursors: this.cursors }
            });
            document.dispatchEvent(event);


        });
    }

    // Method to change cursor programmatically
    setCursor(element, cursorType, fallback = 'auto') {
        if (this.cursors[cursorType]) {
            element.style.cursor = `url('${this.cursors[cursorType]}'), ${fallback}`;
        } else {
            element.style.cursor = fallback;
        }
    }

    // Method to get cursor URL
    getCursor(cursorType) {
        return this.cursors[cursorType] || null;
    }

    // Method to add temporary cursor state
    addTempCursor(element, cursorType, duration = 1000) {
        const originalCursor = element.style.cursor;
        this.setCursor(element, cursorType);

        setTimeout(() => {
            element.style.cursor = originalCursor;
        }, duration);
    }

    // Loading state management
    setLoadingCursor(element, show = true) {
        if (show) {
            this.setCursor(element, 'busy', 'wait');
            element.classList.add('loading-cursor');
        } else {
            element.classList.remove('loading-cursor');
            element.style.cursor = '';
        }
    }

    // Interactive state management
    setInteractiveCursor(element, type = 'link') {
        element.addEventListener('mouseenter', () => {
            this.setCursor(element, type, 'pointer');
        });

        element.addEventListener('mouseleave', () => {
            element.style.cursor = '';
        });
    }
}

// Global cursor loader instance
window.CursorLoader = CursorLoader;

// Auto-initialize when DOM is ready
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', () => {
        window.gamingCursors = new CursorLoader();
    });
} else {
    window.gamingCursors = new CursorLoader();
}

// Export for module systems
if (typeof module !== 'undefined' && module.exports) {
    module.exports = CursorLoader;
}
