// Livewire Error Handler and Page Refresh Manager
// هذا الملف يتعامل مع مشاكل Livewire ويحل مشاكل تحديث الصفحة

window.LivewireErrorHandler = {
    // إعدادات المعالج
    config: {
        maxRetries: 3,
        retryDelay: 2000,
        showUserFriendlyMessages: true,
        autoReload: false,
        debugMode: false
    },

    // عداد المحاولات
    retryCount: 0,

    // حالة الاتصال
    isOnline: true,

    // تهيئة المعالج
    init() {
        this.setupEventListeners();
        this.setupErrorInterception();
        this.setupConnectionMonitoring();
        this.log('Livewire Error Handler initialized');
    },

    // إعداد مستمعات الأحداث
    setupEventListeners() {
        // معالجة الاستجابات
        document.addEventListener('livewire:request', (e) => {
            this.handleRequest(e);
        });

        document.addEventListener('livewire:response', (e) => {
            this.handleResponse(e);
        });

        // معالجة الأخطاء
        document.addEventListener('livewire:error', (e) => {
            this.handleError(e);
        });

        // معالجة حالات الاتصال
        document.addEventListener('livewire:offline', () => {
            this.handleOffline();
        });

        document.addEventListener('livewire:online', () => {
            this.handleOnline();
        });
    },

    // معالجة الطلبات
    handleRequest(event) {
        // إضافة مؤشر التحميل
        this.showLoadingIndicator();
        this.log('Livewire request started', event.detail);
    },

    // معالجة الاستجابات
    handleResponse(event) {
        this.hideLoadingIndicator();

        const response = event.detail.response;

        if (response.status >= 500) {
            this.handleServerError(response);
        } else if (response.status >= 400) {
            this.handleClientError(response);
        } else {
            // إعادة تعيين عداد المحاولات عند نجاح الطلب
            this.retryCount = 0;
        }

        this.log('Livewire response received', response);
    },

    // معالجة الأخطاء
    handleError(event) {
        this.hideLoadingIndicator();
        this.log('Livewire error occurred', event.detail);

        const error = event.detail;

        if (this.shouldRetry()) {
            this.retryRequest();
        } else {
            this.showUserError('حدث خطأ غير متوقع. يرجى تحديث الصفحة.');

            if (this.config.autoReload) {
                setTimeout(() => {
                    window.location.reload();
                }, 3000);
            }
        }
    },

    // معالجة أخطاء الخادم (500+)
    handleServerError(response) {
        this.log('Server error detected', response);

        try {
            const responseText = response.responseText || response.statusText;

            // التحقق من نوع الخطأ
            if (responseText.includes('SQLSTATE') || responseText.includes('Database')) {
                this.showUserError('خطأ في قاعدة البيانات. يرجى المحاولة مرة أخرى.');
            } else if (responseText.includes('CSRF') || responseText.includes('token')) {
                this.handleCSRFError();
            } else if (responseText.includes('Session')) {
                this.handleSessionError();
            } else {
                this.showUserError('خطأ في الخادم. يرجى تحديث الصفحة.');
            }
        } catch (e) {
            this.showUserError('حدث خطأ غير متوقع.');
        }
    },

    // معالجة أخطاء العميل (400-499)
    handleClientError(response) {
        this.log('Client error detected', response);

        if (response.status === 419) {
            this.handleCSRFError();
        } else if (response.status === 401) {
            this.handleAuthError();
        } else {
            this.showUserError('حدث خطأ في الطلب.');
        }
    },

    // معالجة خطأ CSRF
    handleCSRFError() {
        this.showUserError('انتهت صلاحية الجلسة. سيتم تحديث الصفحة...');
        setTimeout(() => {
            window.location.reload();
        }, 2000);
    },

    // معالجة خطأ الجلسة
    handleSessionError() {
        this.showUserError('انتهت الجلسة. يرجى تسجيل الدخول مرة أخرى.');
        setTimeout(() => {
            window.location.href = '/login';
        }, 2000);
    },

    // معالجة خطأ المصادقة
    handleAuthError() {
        this.showUserError('يرجى تسجيل الدخول للمتابعة.');
        setTimeout(() => {
            window.location.href = '/login';
        }, 2000);
    },

    // معالجة قطع الاتصال
    handleOffline() {
        this.isOnline = false;
        this.showUserError('انقطع الاتصال بالإنترنت. جاري المحاولة للاتصال...');
    },

    // معالجة استعادة الاتصال
    handleOnline() {
        this.isOnline = true;
        this.showSuccessMessage('تم استعادة الاتصال بنجاح!');
    },

    // فحص إمكانية إعادة المحاولة
    shouldRetry() {
        return this.retryCount < this.config.maxRetries && this.isOnline;
    },

    // إعادة المحاولة
    retryRequest() {
        this.retryCount++;
        this.log(`Retrying request (attempt ${this.retryCount}/${this.config.maxRetries})`);

        setTimeout(() => {
            window.location.reload();
        }, this.config.retryDelay);
    },

    // إعداد اعتراض الأخطاء
    setupErrorInterception() {
        // اعتراض أخطاء fetch
        const originalFetch = window.fetch;
        window.fetch = async (...args) => {
            try {
                const response = await originalFetch(...args);
                return response;
            } catch (error) {
                this.log('Fetch error intercepted', error);
                throw error;
            }
        };
    },

    // مراقبة الاتصال
    setupConnectionMonitoring() {
        window.addEventListener('online', () => {
            this.handleOnline();
        });

        window.addEventListener('offline', () => {
            this.handleOffline();
        });
    },

    // عرض مؤشر التحميل
    showLoadingIndicator() {
        // يمكن تخصيص هذا حسب تصميم الموقع
        document.body.style.cursor = 'wait';
    },

    // إخفاء مؤشر التحميل
    hideLoadingIndicator() {
        document.body.style.cursor = 'default';
    },

    // عرض رسالة خطأ للمستخدم
    showUserError(message) {
        if (!this.config.showUserFriendlyMessages) return;

        // استخدام SweetAlert إذا كان متوفراً
        if (typeof Swal !== 'undefined') {
            Swal.fire({
                icon: 'error',
                title: 'خطأ',
                text: message,
                confirmButtonText: 'حسناً'
            });
        } else {
            // استخدام alert عادي
            alert(message);
        }
    },

    // عرض رسالة نجاح
    showSuccessMessage(message) {
        if (!this.config.showUserFriendlyMessages) return;

        if (typeof Swal !== 'undefined') {
            Swal.fire({
                icon: 'success',
                title: 'تم بنجاح',
                text: message,
                timer: 2000,
                showConfirmButton: false
            });
        }
    },

    // تسجيل الأحداث (للتطوير)
    log(message, data = null) {
        if (this.config.debugMode) {
            console.log(`[LivewireErrorHandler] ${message}`, data);
        }
    },

    // تفعيل وضع التطوير
    enableDebugMode() {
        this.config.debugMode = true;
        this.log('Debug mode enabled');
    },

    // تعطيل وضع التطوير
    disableDebugMode() {
        this.config.debugMode = false;
    }
};

// تهيئة المعالج عند تحميل الصفحة
document.addEventListener('DOMContentLoaded', () => {
    if (typeof Livewire !== 'undefined') {
        window.LivewireErrorHandler.init();
    }
});

// تصدير للاستخدام العام
window.fixLivewireIssues = function() {
    window.LivewireErrorHandler.retryCount = 0;
    window.location.reload();
};
