<?php

namespace App\Models\Z_Website;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

// notification types: notification, admin_message, charge
//if user_id is null, the notification is for all users
//expired not shown and deleted after 30 days of expiration

class NtfNotification extends Model
{
    protected $fillable = [
        'head',
        'message',
        'link',
        'expires_at',
        'user_id',
        'type',
    ];

    protected $casts = [
        'expires_at' => 'datetime',
        'message' => 'array', // JSON array للترجمة
        'head' => 'array', // JSON array للترجمة
    ];

    /**
     * العلاقة مع جدول حالة المستخدمين
     */
    public function userStatuses(): HasMany
    {
        return $this->hasMany(NtfNotificationUser::class, 'notification_id');
    }

    /**
     * التحقق من انتهاء صلاحية الإشعار
     */
    public function isExpired(): bool
    {
        return $this->expires_at && $this->expires_at->isPast();
    }

    /**
     * التحقق من وجود انتهاء زمني
     */
    public function hasExpiration(): bool
    {
        return !is_null($this->expires_at);
    }

    /**
     * إرسال إشعار للمستخدم مع تحديث الإشعارات
     *
     * @param int|null $userId
     * @param string $type نوع الإشعار (notification, admin_message, charge)
     * @param array $head عنوان الإشعار بعدة لغات
     * @param array $message محتوى الإشعار بعدة لغات
     * @param string|null $link رابط إضافي (اختياري)
     * @param Carbon|null $expiresAt تاريخ انتهاء الإشعار (اختياري)
     * @return self|null
     */
    public static function sendNotificationToUser(
        ?int $userId,
        string $type,
        array $head,
        array $message,
        ?string $link = null,
        ?Carbon $expiresAt = null
    ): ?self {
        try {


            // إنشاء الإشعار
            $notification = self::create([
                'user_id' => $userId,
                'type' => $type,
                'head' => $head,
                'message' => $message,
                'link' => $link,
                'expires_at' => $expiresAt,
            ]);



            Log::info('Notification sent successfully', [
                'notification_id' => $notification->id,
                'user_id' => $userId,
                'type' => $type
            ]);

            return $notification;
        } catch (\Exception $e) {
            Log::error('Failed to send notification: ' . $e->getMessage(), [
                'user_id' => $userId,
                'type' => $type,
                'error_trace' => $e->getTraceAsString()
            ]);
            return null;
        }


    }

//clean Expired notifications
    public static function cleanExpiredNotifications(): void
    {
        try {
            $expiredNotifications = self::whereNotNull('expires_at')
                ->where('expires_at', '<', now()->subDays(30))
                ->get();

            foreach ($expiredNotifications as $notification) {
                // حذف الحالات المرتبطة أولاً
                $notification->userStatuses()->delete();
                // ثم حذف الإشعار نفسه
                $notification->delete();
            }

            Log::info('Expired notifications cleaned successfully', [
                'count' => $expiredNotifications->count()
            ]);
        } catch (\Exception $e) {
            Log::error('Failed to clean expired notifications: ' . $e->getMessage(), [
                'error_trace' => $e->getTraceAsString()
            ]);
        }
    }




}





