<?php

namespace App\Models\Z_Website;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use App\Models\Z_Website\Image;
use Carbon\Carbon;

class Download extends Model
{
    use HasFactory;

    /**
     * The connection name for the model.
     */
    protected $connection = 'sqlsrv';

    /**
     * The table associated with the model.
     */
    protected $table = 'downloads';

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'name',
        'description',
        'link',
        'file_size',
        'version',
        'category',
        'download_count',
        'is_active',
        'image_id',
        'icon',
        'sort_order'
    ];

    /**
     * The attributes that should be cast.
     */
    protected $casts = [
        'is_active' => 'boolean',
        'download_count' => 'integer',
        'sort_order' => 'integer',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    /**
     * Boot the model.
     */
    protected static function boot()
    {
        parent::boot();

        static::creating(function ($download) {
            if (is_null($download->sort_order)) {
                $download->sort_order = static::max('sort_order') + 1;
            }
        });
    }

    /**
     * Get the image associated with the download.
     */
    public function image()
    {
        return $this->belongsTo(Image::class, 'image_id');
    }

    /**
     * Scope to get only active downloads.
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Scope to get downloads by category.
     */
    public function scopeByCategory($query, $category)
    {
        return $query->where('category', $category);
    }

    /**
     * Scope to order downloads by sort order.
     */
    public function scopeOrdered($query)
    {
        return $query->orderBy('sort_order', 'asc')->orderBy('created_at', 'desc');
    }

    /**
     * Increment download count.
     */
    public function incrementDownloadCount()
    {
        $this->increment('download_count');
    }

    /**
     * Format file size for display.
     */
    public function getFormattedFileSizeAttribute()
    {
        if (!$this->file_size) {
            return 'N/A';
        }

        $bytes = floatval($this->file_size);
        $units = ['B', 'KB', 'MB', 'GB', 'TB'];

        for ($i = 0; $bytes > 1024 && $i < count($units) - 1; $i++) {
            $bytes /= 1024;
        }

        return round($bytes, 2) . ' ' . $units[$i];
    }

    /**
     * Get the download categories.
     */
    public static function getCategories()
    {
        return [
            'client' => 'Game Client',
            'patch' => 'Patches & Updates',
            'tools' => 'Tools & Utilities',
            'guide' => 'Guides & Manuals',
            'media' => 'Media & Screenshots'
        ];
    }

    /**
     * Get downloads for the client tab.
     */
    public static function getClientDownloads()
    {
        return static::active()
            ->byCategory('client')
            ->with('image')
            ->ordered()
            ->get();
    }

    /**
     * Get downloads for the tools tab.
     */
    public static function getToolsDownloads()
    {
        return static::active()
            ->byCategory('tools')
            ->with('image')
            ->ordered()
            ->get();
    }
}
