<?php

namespace App\Models\SRO_VT_SHARD;

use Illuminate\Database\Eloquent\Model;
use App\Models\SRO_VT_SHARD\Char;
use App\Models\SRO_VT_SHARD\Items;
use App\Models\SRO_VT_SHARD\_User;

class Inventory extends Model
{
    /**
     * The Database connection name for the model.
     *
     * @var string
     */
    protected $connection = 'sqlsrv_shard';

    /**
     * Indicates if the model should be timestamped.
     *
     * @var bool
     */
    public $timestamps = false;

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'dbo._Inventory';

    /**
     * The table primary Key (composite key)
     * Since this table has a composite key (CharID + Slot),
     * we disable auto-incrementing and set to null
     *
     * @var string
     */
    protected $primaryKey = null;
    public $incrementing = false;

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'CharID',
        'Slot',
        'ItemID'
    ];

    /**
     * The attributes that should be casted to native types.
     *
     * @var array
     */
    protected $casts = [
        'CharID' => 'integer',
        'Slot' => 'integer',
        'ItemID' => 'integer',
    ];

    // =========================================================================
    // RELATIONSHIPS
    // =========================================================================

    /**
     * Get the character that owns this inventory slot
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function character()
    {
        return $this->belongsTo(Char::class, 'CharID', 'CharID');
    }

    /**
     * Get the user that owns this inventory slot (through character)
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function user()
    {
        return $this->belongsTo(_User::class, 'CharID', 'CharID');
    }

    /**
     * Get the item in this inventory slot
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function item()
    {
        return $this->belongsTo(Items::class, 'ItemID', 'ID64');
    }

    // =========================================================================
    // DEPRECATED METHODS (keeping for backward compatibility)
    // =========================================================================

    /**
     * @deprecated Use character() instead
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function getChar()
    {
        return $this->character();
    }

    /**
     * @deprecated Use item() instead
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function getItem()
    {
        return $this->item();
    }

    // =========================================================================
    // HELPER METHODS
    // =========================================================================

    /**
     * Check if this is an equipment slot (0-12)
     *
     * @return bool
     */
    public function isEquipmentSlot()
    {
        return $this->Slot >= 0 && $this->Slot <= 12;
    }

    /**
     * Check if this is an inventory slot (13+)
     *
     * @return bool
     */
    public function isInventorySlot()
    {
        return $this->Slot >= 13;
    }

    /**
     * Get slot name based on slot number
     *
     * @return string
     */
    public function getSlotName()
    {
        $slotNames = [
            0 => 'Head',
            1 => 'Chest',
            2 => 'Shoulder',
            3 => 'Hand',
            4 => 'Legs',
            5 => 'Foot',
            6 => 'Weapon',
            7 => 'Shield',
            8 => 'Job-item',
            9 => 'Earring',
            10 => 'Necklace',
            11 => 'Left-Ring',
            12 => 'Right-Ring'
        ];

        return $slotNames[$this->Slot] ?? "Inventory Slot {$this->Slot}";
    }

    // =========================================================================
    // SCOPES
    // =========================================================================

    /**
     * Scope to get only equipment slots (0-12)
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeEquipment($query)
    {
        return $query->whereBetween('Slot', [0, 12]);
    }

    /**
     * Scope to get only inventory slots (13+)
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeInventoryOnly($query)
    {
        return $query->where('Slot', '>=', 13);
    }

    /**
     * Scope to get items that are not empty (ItemID > 0)
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeWithItems($query)
    {
        return $query->where('ItemID', '>', 0);
    }
}
