<?php

namespace App\Models\SRO_VT_SHARD;

use Illuminate\Database\Eloquent\Model;

class BindingOptionWithItem extends Model
{
    /**
     * The Database connection name for the model.
     *
     * @var string
     */
    protected $connection = 'sqlsrv_shard';

    /**
     * Indicates if the model should be timestamped.
     *
     * @var bool
     */
    public $timestamps = false;

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'dbo._BindingOptionWithItem';

    /**
     * No primary key for this table (composite key scenario)
     *
     * @var string|null
     */
    protected $primaryKey = null;

    /**
     * Indicates if the model has no incrementing primary key
     *
     * @var bool
     */
    public $incrementing = false;

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'nItemDBID',
        'bOptType',
        'nSlot',
        'nOptID',
        'nOptLvl',
        'nOptValue',
        'nParam1',
        'nParam2'
    ];

    /**
     * The attributes that should be casted to native types.
     *
     * @var array
     */
    protected $casts = [
        'nItemDBID' => 'integer',
        'bOptType' => 'integer',
        'nSlot' => 'integer',
        'nOptID' => 'integer',
        'nOptLvl' => 'integer',
        'nOptValue' => 'integer',
        'nParam1' => 'integer',
        'nParam2' => 'integer',
    ];

    // =========================================================================
    // RELATIONSHIPS
    // =========================================================================

    /**
     * Get the item that this binding option belongs to
     * nItemDBID -> Items.ID64
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function item()
    {
        return $this->belongsTo(Items::class, 'nItemDBID', 'ID64');
    }

    // =========================================================================
    // HELPER METHODS
    // =========================================================================

    /**
     * Check if this is an active binding option (nOptValue > 0)
     *
     * @return bool
     */
    public function isActive()
    {
        return $this->nOptValue > 0;
    }

    /**
     * Get the binding option type name
     *
     * @return string
     */
    public function getOptTypeName()
    {
        $types = [
            1 => 'Seal of Star (A)',
            2 => 'Seal of Moon (B)',
            3 => 'Seal of Sun (C)',
            4 => 'Special Seal'
        ];

        return $types[$this->bOptType] ?? 'Unknown Type';
    }

    /**
     * Get formatted binding option display
     *
     * @return string
     */
    public function getFormattedDisplay()
    {
        return sprintf(
            'Slot %d: %s (Lv.%d - Value: %d)',
            $this->nSlot,
            $this->getOptTypeName(),
            $this->nOptLvl,
            $this->nOptValue
        );
    }

    // =========================================================================
    // SCOPES
    // =========================================================================

    /**
     * Scope to get only active binding options (nOptValue > 0)
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeActive($query)
    {
        return $query->where('nOptValue', '>', 0);
    }

    /**
     * Scope to get binding options by type
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @param int $type
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeByType($query, $type)
    {
        return $query->where('bOptType', $type);
    }

    /**
     * Scope to get binding options by slot number
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @param int $slot
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeBySlot($query, $slot)
    {
        return $query->where('nSlot', $slot);
    }

    /**
     * Scope to get binding options for specific item
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @param int $itemId
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeForItem($query, $itemId)
    {
        return $query->where('nItemDBID', $itemId);
    }
}
