<?php

namespace App\Models\SRO_VT_ACCOUNT;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Illuminate\Support\Facades\Hash;

class TbUser extends Authenticatable
{
    use HasFactory, Notifiable;

    /**
     * The connection name for the model.
     */
    protected $connection = 'sqlsrv_account';

    /**
     * The table associated with the model.
     */
    protected $table = 'dbo.TB_User';

    /**
     * The primary key associated with the table.
     */
    protected $primaryKey = 'JID';

    /**
     * Indicates if the model should be timestamped.
     */
    public $timestamps = false;

    /**
     * The attributes that are mass assignable.
     */
    protected $fillable = [
        'StrUserID',
        'Name',
        'password',
        'Email',
        'Status',
        'GMrank',
        'regtime',
        'reg_ip',
    ];

    /**
     * The attributes that should be hidden for serialization.
     */
    protected $hidden = [
        'password',
    ];

    /**
     * Get the attributes that should be cast.
     */
    protected function casts(): array
    {
        return [
            // لا نحتاج password casting مع MD5
        ];
    }

    /**
     * Get the name of the unique identifier for the user.
     */
    public function getAuthIdentifierName(): string
    {
        return 'JID';
    }

    /**
     * Get the unique identifier for the user.
     */
    public function getAuthIdentifier(): mixed
    {
        return $this->{$this->getAuthIdentifierName()};
    }

    /**
     * Get the password for the user.
     */
    public function getAuthPassword(): string
    {
        return $this->password;
    }

    /**
     * Set the password attribute - keep as is for existing MD5 hashes
     */
    public function setPasswordAttribute($value): void
    {
        // حفظ القيمة كما هي - سواء كانت MD5 أو plain text
        $this->attributes['password'] = $value;
    }

    // ==============================================
    // RELATIONSHIPS
    // ==============================================

    /**
     * علاقة مع SK_Silk
     * كل مستخدم له سيلك واحد
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function skSilk()
    {
        return $this->hasOne(\App\Models\SRO_VT_ACCOUNT\SkSilk::class, 'JID', 'JID');
    }

    /**
     * علاقة مع جدول الموقع
     * كل مستخدم له ملف شخصي واحد في الموقع
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function webUser()
    {
        return $this->hasOne(\App\Models\Z_Website\User::class, 'JID', 'JID');
    }

    /**
     * علاقة مع دور المستخدم عبر جدول الموقع
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasOneThrough
     */
    public function role()
    {
        return $this->hasOneThrough(
            \App\Models\Z_Website\UserRole::class,
            \App\Models\Z_Website\User::class,
            'JID', // Foreign key على جدول users
            'id',  // Foreign key على جدول user_roles
            'JID', // Local key على جدول TB_User
            'role_id' // Local key على جدول users
        );
    }

    /**
     * علاقة مع _AccountJID في قاعدة بيانات Shard
     * كل مستخدم له سجل ذهب واحد في كل shard
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function accountJID()
    {
        return $this->hasOne(\App\Models\SRO_VT_SHARD\AccountJID::class, 'JID', 'JID');
    }

    /**
     * علاقة مع _AccountJID عبر StrUserID
     * علاقة بديلة للربط عبر اسم المستخدم
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function accountJIDByUsername()
    {
        return $this->hasOne(\App\Models\SRO_VT_SHARD\AccountJID::class, 'AccountID', 'StrUserID');
    }

    /**
     * Get user initials for UI
     */
    public function initials(): string
    {
        $name = $this->Name ?: $this->StrUserID;
        $words = explode(' ', trim($name));

        if (count($words) >= 2) {
            return strtoupper(substr($words[0], 0, 1) . substr($words[1], 0, 1));
        }

        return strtoupper(substr($name, 0, 2));
    }

    /**
     * Get user's display name
     */
    public function getNameAttribute(): string
    {
        return $this->attributes['Name'] ?: $this->StrUserID;
    }

    /**
     * Get username for login (StrUserID)
     */
    public function getUsernameAttribute(): string
    {
        return $this->StrUserID;
    }

    /**
     * Determine if the user has verified their email address.
     */
    public function hasVerifiedEmail(): bool
    {
        // نعتبر كل المستخدمين verified
        return true;
    }

    /**
     * Mark the given user's email as verified.
     */
    public function markEmailAsVerified(): bool
    {
        // لا نحتاج لعمل شيء
        return true;
    }

    /**
     * تحديد ما إذا كان المستخدم لديه دور معين
     *
     * @param string $roleName
     * @return bool
     */
    public function hasRole(string $roleName): bool
    {
        return $this->role && $this->role->name === $roleName;
    }

    /**
     * تحديد ما إذا كان المستخدم لديه مستوى دور معين أو أعلى
     *
     * @param int $level
     * @return bool
     */
    public function hasRoleLevel(int $level): bool
    {
        return $this->role && $this->role->level >= $level;
    }

    /**
     * الحصول على اسم الدور
     *
     * @return string|null
     */
    public function getRoleName(): ?string
    {
        return $this->role?->name;
    }

    // ==============================================
    // GOLD MANAGEMENT METHODS
    // ==============================================

    /**
     * الحصول على رصيد الذهب
     *
     * @return int
     */
    public function getGold(): int
    {
        return $this->accountJID?->Gold ?? 0;
    }

    /**
     * الحصول على رصيد الذهب مع التنسيق
     *
     * @return string
     */
    public function getFormattedGold(): string
    {
        return $this->accountJID?->formatted_gold ?? '0';
    }

    /**
     * الحصول على رصيد الذهب بصيغة K/M/B
     *
     * @return string
     */
    public function getGoldKFormat(): string
    {
        return $this->accountJID?->gold_k_format ?? '0';
    }

    /**
     * تحديد ما إذا كان لدى المستخدم ذهب كافي
     *
     * @param int $amount
     * @return bool
     */
    public function hasEnoughGold(int $amount): bool
    {
        return $this->accountJID?->hasEnoughGold($amount) ?? false;
    }

    /**
     * إضافة ذهب لحساب المستخدم
     *
     * @param int $amount
     * @return bool
     */
    public function addGold(int $amount): bool
    {
        if (!$this->accountJID) {
            return false;
        }

        return $this->accountJID->addGold($amount);
    }

    /**
     * خصم ذهب من حساب المستخدم
     *
     * @param int $amount
     * @return bool
     */
    public function deductGold(int $amount): bool
    {
        if (!$this->accountJID) {
            return false;
        }

        return $this->accountJID->deductGold($amount);
    }

    /**
     * تعيين كمية ذهب معينة لحساب المستخدم
     *
     * @param int $amount
     * @return bool
     */
    public function setGold(int $amount): bool
    {
        if (!$this->accountJID) {
            return false;
        }

        return $this->accountJID->setGold($amount);
    }
}
