<?php

namespace App\Helpers;

class Sml2Helper
{
    /**
     * Parse SML2 content and convert to HTML
     *
     * @param string $content
     * @return string
     */
    public static function parseToHtml($content)
    {
        if (empty($content)) {
            return '';
        }

        // Remove <sml2> tags
        $content = preg_replace('/<sml2>(.*?)<\/sml2>/s', '$1', $content);

        // Convert font color tags
        // Format: <font color="R,G,B,A">text</font>
        $content = preg_replace_callback(
            '/<font color="([^"]+)">(.*?)<\/font>/i',
            function ($matches) {
                $colorValues = explode(',', $matches[1]);

                if (count($colorValues) >= 3) {
                    $r = intval($colorValues[0]);
                    $g = intval($colorValues[1]);
                    $b = intval($colorValues[2]);
                    $a = isset($colorValues[3]) ? floatval($colorValues[3]) / 255 : 1;

                    $color = "rgba($r, $g, $b, $a)";
                    return '<span style="color: ' . $color . '">' . $matches[2] . '</span>';
                }

                return $matches[2]; // Return text without formatting if color parsing fails
            },
            $content
        );

        // Convert basic HTML tags
        $content = str_replace(['<strong>', '</strong>'], ['<strong class="sml2-strong">', '</strong>'], $content);
        $content = str_replace(['<br>', '<br/>'], '<br class="sml2-br">', $content);

        return $content;
    }

    /**
     * Parse SML2 content and extract plain text
     *
     * @param string $content
     * @return string
     */
    public static function parseToPlainText($content)
    {
        if (empty($content)) {
            return '';
        }

        // Remove all HTML and SML2 tags
        $content = preg_replace('/<sml2>(.*?)<\/sml2>/s', '$1', $content);
        $content = preg_replace('/<font[^>]*>(.*?)<\/font>/i', '$1', $content);
        $content = strip_tags($content);

        // Convert <br> tags to line breaks
        $content = preg_replace('/<br\s*\/?>/i', "\n", $content);

        return trim($content);
    }

    /**
     * Check if content contains SML2 formatting
     *
     * @param string $content
     * @return bool
     */
    public static function hasSml2Formatting($content)
    {
        return !empty($content) &&
               (strpos($content, '<sml2>') !== false ||
                strpos($content, '<font color=') !== false);
    }

    /**
     * Get CSS styles for SML2 rendered content
     *
     * @return string
     */
    public static function getCss()
    {
        return '
        .sml2-content {
            line-height: 1.5;
        }

        .sml2-strong {
            font-weight: 600;
        }

        .sml2-br {
            margin: 0;
            line-height: 1;
        }

        .sml2-content p {
            margin: 0 0 8px 0;
        }

        .sml2-content p:last-child {
            margin-bottom: 0;
        }
        ';
    }

    /**
     * Parse and clean SML2 content for safe HTML output
     *
     * @param string $content
     * @return string
     */
    public static function parseSafe($content)
    {
        $html = self::parseToHtml($content);

        // Additional security: only allow specific tags
        $allowedTags = '<span><strong><br><p><em><i><b>';

        return strip_tags($html, $allowedTags);
    }
}
