<?php

namespace App\Helpers;

use App\Models\Z_Website\SiteSettings;
use Illuminate\Support\Facades\Cache;

class SiteSettingsHelper
{
    /**
     * Get site settings with caching
     *
     * @return array
     */
    public static function getSettings()
    {
        return Cache::remember('site_settings', 300, function () {
            $settings = SiteSettings::first();
            return $settings ? $settings->settings : [];
        });
    }

    /**
     * Get a specific setting value
     *
     * @param string $key
     * @param mixed $default
     * @return mixed
     */
    public static function get($key, $default = null)
    {
        $settings = self::getSettings();
        return $settings[$key] ?? $default;
    }

    /**
     * Get social media links
     *
     * @return array
     */
    public static function getSocialLinks()
    {
        return [
            'facebook' => self::get('facebook_url', '#'),
            'youtube' => self::get('youtube_url', '#'),
            'discord_id' => self::get('discord_id', ''),
            'discord_invite' => self::get('discord_invite', ''),
            'contact_email' => self::get('contact_email', 'support@4tressro.com'),
        ];
    }

    /**
     * Get server rates information
     *
     * @return array
     */
    public static function getServerRates()
    {
        return [
            'exp' => self::get('sro_exp', '45'),
            'gold' => self::get('sro_exp_gold', '45'),
            'drop' => self::get('sro_exp_drop', '45'),
            'job' => self::get('sro_exp_job', '40'),
            'party' => self::get('sro_exp_party', '55'),
        ];
    }

    /**
     * Get server limits and capacities
     *
     * @return array
     */
    public static function getServerLimits()
    {
        return [
            'max_server' => self::get('sro_max_server', '1000'),
            'level_cap' => self::get('sro_cap', '130'),
            'ip_limit' => self::get('sro_ip_limit', '99'),
            'hwid_limit' => self::get('sro_hwid_limit', '8'),
        ];
    }

    /**
     * Get currency names
     *
     * @return array
     */
    public static function getCurrencyNames()
    {
        return [
            'silk' => self::get('sro_silk_name', 'Silk'),
            'silk_gift' => self::get('sro_silk_gift_name', '4T Donate'),
        ];
    }

    /**
     * Get fortress status
     *
     * @return array
     */
    public static function getFortressStatus()
    {
        return [
            'jangan' => self::get('jangan_fortress', true),
            'bandit' => self::get('bandit_fortress', true),
            'hotan' => self::get('hotan_fortress', false),
            'constantinople' => self::get('constantinople_fortress', true),
        ];
    }

    /**
     * Get ranking system status
     *
     * @return array
     */
    public static function getRankingStatus()
    {
        return [
            'char_ranking' => self::get('char_ranking', true),
            'guild_ranking' => self::get('guild_ranking', true),
            'job_ranking' => self::get('job_ranking', true),
            'trader_ranking' => self::get('trader_ranking', true),
            'hunter_ranking' => self::get('hunter_ranking', true),
            'thief_ranking' => self::get('thief_ranking', true),
            'unique_ranking' => self::get('unique_ranking', true),
            'free_pvp_ranking' => self::get('free_pvp_ranking', true),
            'job_pvp_ranking' => self::get('job_pvp_ranking', true),
        ];
    }

    /**
     * Check if registration is open
     *
     * @return bool
     */
    public static function isRegistrationOpen()
    {
        return !self::get('registration_close', false);
    }

    /**
     * Clear settings cache
     *
     * @return void
     */
    public static function clearCache()
    {
        Cache::forget('site_settings');
    }

    /**
     * Get all formatted settings for quick access
     *
     * @return array
     */
    public static function getAllFormatted()
    {
        return [
            'social' => self::getSocialLinks(),
            'rates' => self::getServerRates(),
            'limits' => self::getServerLimits(),
            'currency' => self::getCurrencyNames(),
            'fortress' => self::getFortressStatus(),
            'ranking' => self::getRankingStatus(),
            'registration_open' => self::isRegistrationOpen(),
        ];
    }
}
