<?php

if (!function_exists('active_class')) {
    /**
     * Get active class if current route matches
     *
     * @param string|array $routes
     * @param string $class
     * @return string
     */
    function active_class($routes, $class = 'active')
    {
        if (is_array($routes)) {
            foreach ($routes as $route) {
                if (request()->routeIs($route)) {
                    return $class;
                }
            }
        } else {
            if (request()->routeIs($routes)) {
                return $class;
            }
        }

        return '';
    }
}

if (!function_exists('is_active_route')) {
    /**
     * Check if current route matches
     *
     * @param string|array $routes
     * @return bool
     */
    function is_active_route($routes)
    {
        if (is_array($routes)) {
            foreach ($routes as $route) {
                if (request()->routeIs($route)) {
                    return true;
                }
            }
        } else {
            return request()->routeIs($routes);
        }

        return false;
    }
}

if (!function_exists('active_url')) {
    /**
     * Get active class if current URL matches pattern
     *
     * @param string $pattern
     * @param string $class
     * @return string
     */
    function active_url($pattern, $class = 'active')
    {
        $currentPath = request()->path();

        // Handle wildcard patterns
        if (str_contains($pattern, '*')) {
            $pattern = str_replace('*', '.*', $pattern);
            return preg_match('#^' . $pattern . '$#', $currentPath) ? $class : '';
        }

        // Handle exact patterns
        if (str_contains($pattern, '/')) {
            return request()->is($pattern) ? $class : '';
        }

        // Handle simple patterns
        return str_contains($currentPath, $pattern) ? $class : '';
    }
}

if (!function_exists('is_active_url')) {
    /**
     * Check if current URL matches pattern
     *
     * @param string $pattern
     * @return bool
     */
    function is_active_url($pattern)
    {
        $currentPath = request()->path();

        // Handle wildcard patterns
        if (str_contains($pattern, '*')) {
            $pattern = str_replace('*', '.*', $pattern);
            return (bool) preg_match('#^' . $pattern . '$#', $currentPath);
        }

        // Handle exact patterns
        if (str_contains($pattern, '/')) {
            return request()->is($pattern);
        }

        // Handle simple patterns
        return str_contains($currentPath, $pattern);
    }
}

if (!function_exists('active_hash')) {
    /**
     * Get active class if current URL hash matches pattern
     *
     * @param string $hash
     * @param string $class
     * @return string
     */
    function active_hash($hash, $class = 'active')
    {
        if (!isset($_SERVER['HTTP_REFERER'])) {
            return '';
        }

        $referer = parse_url($_SERVER['HTTP_REFERER'], PHP_URL_FRAGMENT);
        return $referer === ltrim($hash, '#') ? $class : '';
    }
}

if (!function_exists('is_active_hash')) {
    /**
     * Check if current URL hash matches pattern
     *
     * @param string $hash
     * @return bool
     */
    function is_active_hash($hash)
    {
        if (!isset($_SERVER['HTTP_REFERER'])) {
            return false;
        }

        $referer = parse_url($_SERVER['HTTP_REFERER'], PHP_URL_FRAGMENT);
        return $referer === ltrim($hash, '#');
    }
}

if (!function_exists('get_navigation_data')) {
    /**
     * Get navigation data from JSON file
     *
     * @return array
     */
    function get_navigation_data()
    {
        static $navigationData = null;

        if ($navigationData === null) {
            $filePath = resource_path('data/navigation.json');
            if (file_exists($filePath)) {
                $navigationData = json_decode(file_get_contents($filePath), true);
            } else {
                // Fallback navigation data
                $navigationData = [
                    'navigation' => [
                        'primary' => [
                            ['name' => 'home', 'route' => 'home', 'icon' => 'fas fa-home', 'label' => 'nav.home', 'title' => 'Home'],
                            ['name' => 'download', 'route' => 'download', 'icon' => 'fas fa-download', 'label' => 'nav.download', 'title' => 'Download'],
                            ['name' => 'rank', 'route' => 'rank', 'icon' => 'fas fa-trophy', 'label' => 'nav.ranking', 'title' => 'Ranking']
                        ],
                        'dropdown' => [
                            ['name' => 'about', 'route' => 'about', 'icon' => 'fas fa-book', 'label' => 'nav.aboutGame', 'title' => 'About Game'],
                            ['name' => 'news', 'route' => 'news', 'icon' => 'fas fa-newspaper', 'label' => 'nav.newsEvents', 'title' => 'News & Events'],
                            ['name' => 'shop', 'route' => 'shop', 'icon' => 'fas fa-store', 'label' => 'nav.itemShop', 'title' => 'Item Shop'],
                            ['name' => 'community', 'route' => 'community', 'icon' => 'fas fa-users', 'label' => 'nav.community', 'title' => 'Community'],
                            ['name' => 'support', 'route' => 'support', 'icon' => 'fas fa-headset', 'label' => 'nav.support', 'title' => 'Support'],
                            ['name' => 'guides', 'route' => 'guides', 'icon' => 'fas fa-book-open', 'label' => 'nav.gameGuides', 'title' => 'Game Guides'],
                            ['name' => 'gallery', 'route' => 'gallery', 'icon' => 'fas fa-images', 'label' => 'nav.mediaGallery', 'title' => 'Media Gallery']
                        ],
                        'user_menu' => [
                            ['name' => 'dashboard', 'route' => 'dashboard', 'icon' => 'fas fa-tachometer-alt', 'label' => 'user.dashboard', 'title' => 'Dashboard']
                        ]
                    ],
                    'dropdown_trigger' => [
                        'icon' => 'fas fa-th-large',
                        'label' => 'nav.more',
                        'title' => 'More'
                    ]
                ];
            }
        }

        return $navigationData;
    }
}
