<?php

namespace App\Exceptions;

use Illuminate\Foundation\Exceptions\Handler as ExceptionHandler;
use Illuminate\Support\Facades\Log;
use Throwable;

class Handler extends ExceptionHandler
{
    /**
     * The list of the inputs that are never flashed to the session on validation exceptions.
     *
     * @var array<int, string>
     */
    protected $dontFlash = [
        'current_password',
        'password',
        'password_confirmation',
    ];

    /**
     * Register the exception handling callbacks for the application.
     */
    public function register(): void
    {
        $this->reportable(function (Throwable $e) {
            // معالجة خاصة لأخطاء Livewire
            if (request()->header('X-Livewire') || str_contains(request()->url(), '/livewire/update')) {
                Log::channel('livewire')->error('Livewire Error', [
                    'message' => $e->getMessage(),
                    'file' => $e->getFile(),
                    'line' => $e->getLine(),
                    'url' => request()->fullUrl(),
                    'user_id' => \Illuminate\Support\Facades\Auth::check() ? \Illuminate\Support\Facades\Auth::user()->id : null,
                    'trace' => $e->getTraceAsString()
                ]);
            }
        });
    }

    /**
     * Render an exception into an HTTP response.
     */
    public function render($request, Throwable $e)
    {
        // معالجة خاصة لطلبات Livewire
        if ($request->header('X-Livewire') || str_contains($request->url(), '/livewire/update')) {

            // إذا كان خطأ في قاعدة البيانات
            if ($e instanceof \Illuminate\Database\QueryException) {
                Log::error('Livewire Database Error', [
                    'message' => $e->getMessage(),
                    'sql' => $e->getSql() ?? 'SQL not available',
                    'bindings' => $e->getBindings() ?? [],
                    'url' => $request->fullUrl()
                ]);

                return response()->json([
                    'message' => 'حدث خطأ في قاعدة البيانات. يرجى المحاولة مرة أخرى.',
                    'error_type' => 'database'
                ], 500);
            }

            // إذا كان خطأ في التشفير
            if ($e instanceof \Illuminate\Encryption\MissingAppKeyException) {
                Log::error('Livewire Encryption Error', [
                    'message' => $e->getMessage(),
                    'url' => $request->fullUrl()
                ]);

                return response()->json([
                    'message' => 'خطأ في تكوين الخادم. يرجى التواصل مع المطور.',
                    'error_type' => 'encryption'
                ], 500);
            }

            // للأخطاء الأخرى
            return response()->json([
                'message' => 'حدث خطأ غير متوقع. يرجى تحديث الصفحة والمحاولة مرة أخرى.',
                'error_type' => 'general'
            ], 500);
        }

        return parent::render($request, $e);
    }
}
